<?php
/*
 * This file is part of the DkLib.
 *   You can redistribute it and/or modify
 *   it under the terms of the GNU Lesser General Public License as published by
 *   the Free Software Foundation, either version 3 of the License, or
 *   (at your option) any later version.
 *
 *   You should have received a copy of the GNU Lesser General Public License
 *   If not, see <http://www.gnu.org/licenses/>.
 *
 * @author Michael Mifsud
 * @author Darryl Ross
 * @link http://www.tropotek.com/
 * @license Copyright 2007 Michael Mifsud
 */

/**
 * A PHP5 DOM Template Library
 * 
 * 
 * TODO: Add documentation on how to use this lib
 * 
 * 
 * 
 * NOTE: Var attribute values cannot begin with '__' because they are reserved for the 
 *   template system's internal functions.
 * 
 * @package Dom
 */
class Dom_Template
{
    
    /**
     * @var DOMDocument
     */
    protected $document = null;
    
    /**
     * @var Dom_Template
     */
    protected $parent = null;
    
    
    
    
    /**
     * An array of var DOMElement objects
     * @var array 
     */
    protected $var = array();
    
    /**
     * An array of choice DOMElement objects
     * @var array 
     */
    protected $choice = array();
    
    /**
     * An array of repeat DOMElement objects
     * @var array 
     */
    protected $repeat = array();
    
    /**
     * An array of form DOMElement objects
     * @var DOMElement
     */
    protected $form = array();
    
    /**
     * An array of formElement DOMElement objects
     * @var array
     */
    protected $formElement = array();
    
    /**
     * Header elements to be added
     * @var array
     */
    protected $headers = array();
    
    /**
     * Comment tags to be removed
     * @var array
     */
    protected $comments = array();
    
    /**
     * An array of DOMElement objects
     * @var array
     */
    protected $nextSibling = array();
    
    
    /**
     * The head tag of a html page
     * @var DOMElement
     */
    protected $head = null;
    
    /**
     * The body tag of a html page
     * @var DOMElement
     */
    protected $body = null;
    
    /**
     * The head tag of a html page
     * @var DOMElement
     */
    protected $title = null;
    
    
    
    
    /**
     * Set to true if this template has been parsed
     * @var boolean
     */
    protected $parsed = false;
    
    
    
    /**
     * The constructor
     *
     * @param DOMDocument $doc
     */
    function __construct(DOMDocument $doc)
    {
        $this->document = $doc;
        $this->init($this->document->documentElement);
    }

    
    
    /**
     * Make a template from an xml file
     * 
     * @param string $xmlFile
     * @return Dom_Template
     * @throws RuntimeException If there is an error with the XML template
     */
    static function load($xmlFile)
    {
        if (!is_file($xmlFile)) {
            return null;
        }
        $xmlStr = file_get_contents($xmlFile);
        $doc = new DOMDocument();
        if (!$doc->loadXML(self::cleanXml($xmlStr))) {
            throw new RuntimeException('Cannot create `Template` error reading xml.');
        }
        return new Dom_Template($doc);
    }

    /**
     * Make a template from an xml string
     * 
     * @param string $xmlStr
     * @return Dom_Template
     * @throws RuntimeException If there is an error with the XML template
     */
    static function loadXml($xmlStr)
    {
        if ($xmlStr == '') {
            return null;
        }
        $doc = new DOMDocument();
        if (!$doc->loadXML(self::cleanXml($xmlStr))) {
            throw new RuntimeException('Cannot create `Template` error reading xml.');
        }
        return new Dom_Template($doc);
    }
    
    /**
     * Get the xml and return the cleaned string
     * A good place to clean any nasty html entities and other non valid XML/XHTML elements
     *  
     * @param string $xml
     * @return string
     */
    static function cleanXml($xml)
    {
        $xml = str_replace(array('&nbsp;', '<br>', '<BR>'), array('&#160;', '<br/>', '<br/>'), $xml);
        return $xml;
    }
    
    
    /**
     * A private method to initalise the template.
     * 
     * @param DOMElement $node 
     * @param string $form
     */
    private function init($node, $form = '') 
    {
    
        if ($this->isParsed()) {
            return;
        }
        if ($node->nodeType == XML_ELEMENT_NODE) {
        
            // Do any subclass post initalisations
            // Be sure to not remove attributes that are required for the template
            if ($this->postInit($node, $form)) {
                return;
            }
        	
            // Store all repeat regions
            if ($node->hasAttribute('repeat')) {
                $this->repeat[$node->getAttribute('repeat')] = $node;
                $node->removeAttribute('repeat');
                return;
            }
            
            // Store all var nodes
            if ($node->hasAttribute('var')) {
                $var = $node->getAttribute('var');
                if (!array_key_exists($var, $this->var)) {
                    $this->var[$var] = array();
                }
                $this->var[$var][] = $node;
                $node->removeAttribute('var');
            }
            
            // Store all choice nodes
            if ($node->hasAttribute('choice')) {
                if (!array_key_exists($node->getAttribute('choice'), $this->choice)) {
                    $this->choice[$node->getAttribute('choice')] = array();
                    $this->choice[$node->getAttribute('choice')]['node'] = array();
                    $this->choice[$node->getAttribute('choice')]['set'] = false;
                }
                $this->choice[$node->getAttribute('choice')]['node'][] = $node;
                $node->removeAttribute('choice');
            }
            
            // Store all Form nodes
            if ($node->nodeName == 'form') {
                $form = $node->getAttribute('id');
                if ($form == null) {
                    $form = $node->getAttribute('name');
                }
                $this->formElement[$form] = array();
                $this->form[$form] = $node;
            }
            
            // Store all FormElement nodes
            if ($node->nodeName == 'input' || $node->nodeName == 'textarea' || 
                $node->nodeName == 'select') 
            {
                $id = $node->getAttribute('name');
                if ($id == null) {
                    $id = $node->getAttribute('id');
                }
                if (!isset($this->formElement[$form][$id])) {
                    $this->formElement[$form][$id] = array();
                }
                $this->formElement[$form][$id][] = $node;
            }
            
            if ($node->nodeName == 'title') {
                $this->title = $node;
            }
            if ($node->nodeName == 'head') {
                $this->head = $node;
            }
            if ($node->nodeName == 'body') {
                $this->body = $node;
            }
            
            // iterate through the elements
            $children = $node->childNodes;
            foreach ($children as $child) {
                if ($child->nodeType == XML_COMMENT_NODE) {
                    $this->comments[] = $child;
                }
                $this->init($child, $form);
            }
            $form = '';
        }
    }
    
    /**
     * This function is used when subclassing the template 
     * implement this function to capture new nodes.
     * 
     * @param DOMElement $node
     * @param string $form The form name if inside a form.
     */
    protected function postInit(DOMElement $node, $form = '') { }
    
    
    
    /**
     * Replace the text of one or more var nodes
     * 
     * @param string $var The var's name.
     * @param string $value The vars value inside the tags.
     */
    function replaceText($var, $value) 
    {
        if (!$this->keyExists('var', $var)) {
            return;
        }
        $nodes = $this->var[$var];
        if ($nodes == null) {
            return;
        }
        
        foreach ($nodes as $node) {
            $this->removeChildren($node);
            if (is_object($value)) {
                $newNode = $this->document->createTextNode(self::objectToString($value));
            } else {
                $newNode = $this->document->createTextNode($value);
            }
            $node->appendChild($newNode);
        }
    }
    
    /**
     * Get the text inside a var node.
     * 
     * @param string $var 
     * @return string 
     */
    function getText($var) 
    {
        if (!$this->keyExists('var', $var)) {
            return '';
        }
        return $this->var[$var][0]->nodeValue;
    }
    
    
        
    /**
     * Replace an attribute value.
     * 
     * @param string $var 
     * @param string $attr
     * @param string $value 
     */
    function replaceAttr($var, $attr, $value) 
    {
        if (!$this->keyExists('var', $var)) {
            return;
        }
        $nodes = $this->var[$var];
        if ($nodes == null) {
            return;
        }
        
        foreach ($nodes as $node) {
            if ($value == null) {
                $node->removeAttribute($attr);
            }else{
                $node->setAttribute($attr, self::objectToString($value));
            }
        }
    }
    
    /**
     * Retreive the text contained within an attribute of a node.
     * 
     * @param string $var
     * @param string $attr 
     * @return string 
     */
    function getAttr($var, $attr) 
    {
        if (!$this->varExists($var)) {
            return '';
        }
        return $this->var[$var][0]->getAttribute($attr);
    }
    
    /**
     * Set a choice node to become visible in a document.
     * 
     * @param string $choice The name of the choice
     */
    function setChoice($choice) 
    {
        if (!$this->keyExists('choice', $choice)) {
            return;
        }
        $this->choice[$choice]['set'] = true;
    }
    
    /**
     * Set a choice node to become invisible in a document.
     * 
     * @param string $choice The name of the choice
     */
    function unsetChoice($choice) 
    {
        if (!$this->keyExists('choice', $choice)) {
            return;
        }
        $this->choice[$choice]['set'] = false;
    }
    
    /**
     * Return a form object from the document.
     * 
     * @param string $id
     * @return Dom_Form 
     */
    function getForm($id = '') 
    {
        if (!$this->keyExists('form', $id)) {
            return;
        }
        return new Dom_Form($this->form[$id], $this->formElement[$id], $this);
    }
    
    /**
     * Get a repeating region from a document.
     * 
     * @param string $repeat 
     * @return Dom_Repeat 
     */
    function getRepeat($repeat) 
    {
        if (!$this->keyExists('repeat', $repeat)) {
            return;
        }
        $repeat = new Dom_Repeat($repeat, $this);
        return $repeat;
    }
    
    
    
    
    
    

    
    /**
     * Get a var element node from the document.
     * 
     * @param string $var
     * @return DOMElement
     */
    function getVarElement($var) 
    {
        if (!$this->keyExists('var', $var)) {
            return;
        }
        return $this->var[$var][0];
    }
    
    /**
     * Get a repeat element node from the document.
     * 
     * @param string $repeat
     * @return DOMElement
     */
    function getRepeatElement($repeat) 
    {
        if (!$this->keyExists('repeat', $repeat)) {
            return;
        }
        return $this->repeat[$repeat];
    }
    
    /**
     * Get a DOMElement from the document based on its unique ID
     * ID attributes should be unique for XHTML documents, multiple names
     * are ignored and only the first node found is returned.
     *
     * @param string $id
     * @return DOMElement Returns null if not found
     */
    function getElementById($id)
    {
        $this->document->getElementById($id);
    }
    
    /**
     * Return the current list of header nodes
     *
     * @return array
     */
    function getHeaderList()
    {
        return $this->headers;
    }
    
    /**
     * Return the body node.
     * 
     * @return DOMElement
     */
    function getBodyElement() 
    {
        return $this->body;
    }
    
    /**
     * Gets the page title text.
     * 
     * @return string The title.
     */
    function getTitleText()
    {
        return $this->title->nodeValue;
    }

    /**
     * Sets the document title text if available.
     * 
     * @param string The title.
     */
    function setTitleText($value)
    {
        if ($this->title == null) {
            throw new Exception('This document has no title node.');
        }
        $this->removeChildren($this->title);
        $this->title->nodeValue = self::objectToString($value);
    }

    
    /**
     * Check if a repeat,choice,var,form (template property) Exists.
     * 
     * @param string $property 
     * @param string $key 
     */
    protected function keyExists($property, $key) 
    {
        if (!array_key_exists($key, $this->$property)) {
            return false;
        }
        return true;
    }
    

    
    /**
     * Appends an element to the widgets of the HTML head element.
     *
     * In the form of:
     *  <$elementName $attributes[$key]="$attributes[$key].$value">$value</$elementName>
     *
     * NOTE: Only allows unique headers. An md5 hash is refrenced from all input parameters.
     *  Any duplicate headers are discarded.
     * 
     * @param string $elementName
     * @param array $attributes An associative array of (attr, value) pairs.
     * @param string $value The element value.
     */
    function appendHeadElement($elementName, $attributes, $value = '') 
    {
        if ($this->head == null) {
            $this->reportError('This template has no head node.');
        }
        $preKey = $elementName.$value;
        foreach ($attributes as $k => $v) {
            $preKey .= $k.$v;
        }
        $hash = md5($preKey);

        if (!array_key_exists($hash, $this->headers)) {
            $this->headers[$hash]['elementName'] = $elementName;
            $this->headers[$hash]['attributes'] = $attributes;
            $this->headers[$hash]['value'] = self::objectToString($value);
        }
    }
    
    
    
    /**
     * Insert some HTML formatted text into a var node.
     *
     * @param string $var 
     * @param string $html 
     * @param string $encoding
     * @return boolean Returns true on success 
     */
    function replaceHTML($var, $html, $encoding = 'UTF-8') 
    {
        if ($html == null) {
            return false;
        }
        if (!$this->keyExists('var', $var)) {
            return false;
        }
        $nodes = $this->var[$var];
        foreach ($nodes as $node) {
            self::insertHTML($node, $html, $encoding);
        }
        return true;
    }
    
    /**
     * Insert some HTML formatted text into a dom node.
     *
     * @param DOMElement $element 
     * @param string $html 
     * @param string $encoding 
     * @return boolean Returns true on success 
     */
    static function insertHTML($element, $html, $encoding = 'UTF-8') 
    {
        if ($html == null) {
            return false;
        }
        $elementDoc = $element->ownerDocument;
        
        $html = mb_convert_encoding($html, $encoding, 'UTF-8');
        $nameStr = '<'.$element->nodeName;
        foreach ($element->attributes as $v) {
        	$nameStr .= sprintf(' %s="%s"', $v->name, $v->value);
        }
        $html = $nameStr . '>' . $html . '</' . $element->nodeName . '>';
        
        $html = sprintf('<html><head><meta http-equiv="Content-Type" content="text/html; charset=%s"></head><body><div>%s</div></body></html>', $encoding, $html);
        $doc = @DOMDocument::loadHTML($html);
        
        $list = $doc->getElementsByTagName('div');
        $contentNode = $list->item(0);
        if ($contentNode == null) {
            return false;
        }
        $contentNode = $contentNode->firstChild;
        $contentNode = $elementDoc->importNode($contentNode, true);
        $element->parentNode->replaceChild($contentNode, $element);
        return true;
    }
    
    /**
     * Insert a doc into a var in this objects doc
     * The DOMDocument will replace the var tag
     * 
     * @param string $var 
     * @param DOMDocument $doc 
     * @return boolean Returns true on success 
     */
    function insertDoc($var, $doc) 
    {
        if (!$this->keyExists('var', $var)) {
            return false;
        }
        $newNode = $this->document->importNode($doc->documentElement, true);
        foreach ($this->var[$var] as $node) {
            $node->parentNode->replaceChild($newNode, $node);
            $this->removeChildren($node);
        }
        return true;
    }
    
    /**
     * Insert a template into a var in this objects doc
     * The DOMDocument will replace the var tag
     * 
     * This will also grab any headers in the supplied template.
     * 
     * @param string $var 
     * @param Dom_Template $template 
     * @return boolean Returns true on success 
     */
    function insertTemplate($var, $template) 
    {
        if (!$this->keyExists('var', $var)) {
            return false;
        }
        $doc = $template->getDocument();
        $this->headers = array_merge($this->headers, $template->getHeaderList());
        return $this->insertDoc($var, $doc);
    }
    
    /**
     * Append documents before the $var element
     * 
     * @param string $var
     * @param DOMDocument $doc
     * @return boolean Returns true on success 
     */
    function appendDoc($var, $doc) 
    {
        if (!$this->keyExists('var', $var)) {
            return false;
        }
        foreach ($this->var[$var] as $el) {
            $node = $this->document->importNode($doc->documentElement, true);
            $el->parentNode->insertBefore($node, $el);
        }
        return true;
    }
    
    /**
     * Append a template before the $var element
     * 
     * This wil also grab any headers in the $template.
     * 
     * @param string $var
     * @param Dom_Template $template
     * @return boolean Returns true on success 
     */
    function appendTemplate($var, $template) 
    {
        if (!$this->keyExists('var', $var)) {
            return false;
        }
        $doc = $template->getDocument();
        $this->headers = array_merge($this->headers, $template->getHeaderList());
        return $this->appendDoc($var, $doc);
    }
    
    
    
    
    /**
     * Get the parsed state of the template.
     * If true then no more changes can be made to the template
     * 
     * @return boolean
     */
    function isParsed()
    {
        return $this->parsed;
    }
    
    /**
     * Return a parsed Dom document.
     * After using this call you can no longer use the template render functions
     * as no changes will be made to the template unless you use DOM functions
     * 
     * @param boolean $parse Set to false to avoid parsing and return DOMDocument in its current state
     * @return DOMDocument 
     */
    function getDocument($parse = true)
    {
        if (!$this->isParsed() && $parse) {
            foreach ($this->comments as $node) {
                if ($node->ownerDocument != null && $node != null && $node->parentNode != null && 
                    $node->parentNode->nodeName != 'script' && $node->parentNode->nodeName != 'style') 
                {
                   $node->parentNode->removeChild($node);
                }
            }
            
            foreach ($this->repeat as $name => $node) {
                $node->parentNode->removeChild($node);
                unset($this->repeat[$name]);
            }
            foreach ($this->choice as $name => $nodes) {
                if (!$nodes['set']) {
                    foreach ($nodes['node'] as $node) {
                        if ($node != null && $node->parentNode != null ) {
                           $node->parentNode->removeChild($node);
                        }
                    }
                }
                unset($this->choice[$name]);
            }
            
            if ($this->head != null) {
                $hookNode = null;
                if ($this->title != null && $this->title->nextSibling != null) {
                    $hookNode = $this->title->nextSibling;
                }
                if ($hookNode == null && $this->head->firstChild != null) {
                    $hookNode = $this->head->firstChild;
                }
                $ordered = array();
                $meta = array();
                $other = array();
                foreach ($this->headers as $header) {
                    if ($header['elementName'] == 'meta') {
                        $meta[] = $header;
                    } else {
                        $other[] = $header;
                    }
                }
                $ordered = array_merge($meta, $other);
                foreach ($ordered as $header) {
                    // Insert into template
                    $node = $this->document->createElement($header['elementName']);
                    if ($header['value'] != null) {
                        if (strtolower($header['elementName']) == 'script') {  // Javascript/css hack
                            $cm = $this->document->createTextNode("//");
                            $ct = $this->document->createCDATASection("\n" . $header['value'] . "\n//");
                            $node->appendChild($cm);
                            $node->appendChild($ct);
                        } else {
                            $ct = $this->document->createCDATASection($header['value']);
                            $node->appendChild($ct);
                        }
                    }
                    $nl = $this->document->createTextNode("\n");
                    foreach ($header['attributes'] as $k => $v) {
                        $node->setAttribute($k, self::objectToString($v));
                    }
                    if ($hookNode != null) {
                        $this->head->insertBefore($nl, $hookNode);
                        $this->head->insertBefore($node, $hookNode);
                    } else {
                        $this->head->appendChild($nl);
                        $this->head->appendChild($node);
                    }
                }
            }
            
            $this->parsed = true;
        }
        return $this->document;
    }
    
    /**
     * Removes all children from a node.
     * 
     * @param DOMNode $node
     */
    protected function removeChildren($node)
    {
        if ($node == null) {
            throw new Exception('Null node value. This is usually caused by invalid XHTML/XML in a template.');
        }
        while ($node->hasChildNodes()) {
            $node->removeChild($node->childNodes->item(0));
        }
    }
    
    /**
     * ??? Used by the repeat region.
     * 
     * @param string $var 
     * @param DOMElement $nextSibling 
     * @todo Add to the Repeat object
     */
    protected function setNextSibling($var, $nextSibling) 
    {
        if (!array_key_exists($var, $this->nextSibling)) {
            $this->nextSibling[$var] = $nextSibling;
        }
    }
    
    /**
     * ?? Used by the repeat region.
     * 
     * @param string $var
     * @return DOMElement next sibling
     * @todo Add to the Repeat object
     */
    protected function getNextSibling($var) 
    {
        return $this->nextSibling[$var];
    }
    
    /**
     * Return a string from an object.
     * 
     * @param mixed $obj
     */
    static function objectToString($obj) 
    {
        if (is_object($obj) && method_exists($obj, 'toString')) {
            return $obj->toString();
        } else if (is_object($obj) && method_exists($obj, '__toString')) {
            return $obj->toString();
        } else {
            return $obj;
        }
    }
    
    /**
     * Recive the document in the format of 'xml' or 'html'.
     * NOTE: This method does not parse the template document.
     * 
     * @param string $type 'xml' or 'html' 
     * @return string 
     */
    function toString($type = 'xml') 
    {
        if ($type == 'xml') {
            return $this->document->saveXML();
        }else{
            return $this->document->saveHTML();
        }
    }
    function __toString()
    {
        return $this->toString();
    }
    
}


/**
 * A repeat region is a sub template of a parent templates nodes.
 * 
 * @package Dom
 */
class Dom_Repeat extends Dom_Template 
{
    
    /**
     * @var DOMElement
     */
    protected $repeatNode = null;
    
    /**
     * @var string 
     */
    protected $repeatName ='';
    
    
    /**
     * __construct
     * 
     * @param string $repeatName
     * @param Dom_Template $parent 
     * @todo It would be good to send the repeat node as a parameter 
     *       but we need access to the node name, ideas??
     */
    function __construct($repeatName, Dom_Template $parent) 
    {
        $repeatNode = $parent->getRepeatElement($repeatName);
        $doc = new DOMDocument();
        $node = $doc->importNode($repeatNode, true);
        $doc->appendChild($node);
        parent::__construct($doc);
        $this->repeatNode = $repeatNode;
        $this->parent = $parent;
        $this->repeatName = $repeatName;
        $this->parent->setNextSibling('__'.$this->repeatName, $repeatNode->nextSibling);
    }
    
    /**
     * Append a repeating region to the document.
     * Repeating regions are appended to the supplied var.
     * If the var is null or '' then the repeating region is appended
     * to is original location in the parent template.
     * 
     * @param string $var
     */
    function append($var = '') 
    {
        if ($var != '') {
            $appendNode = $this->parent->getVarElement($var);
            $this->parent->setNextSibling($var, $appendNode->nextSibling);
        } else {
            $appendNode = $this->repeatNode;
            $var = '__'.$this->repeatName;
        }
        $parentDoc = $appendNode->ownerDocument;
        // Clone parsed node
        $clone = $parentDoc->importNode($this->getDocument()->documentElement, true);
        if ($this->parent->getNextSibling($var) != null) {
            $appendNode->parentNode->insertBefore($clone, $this->parent->getNextSibling($var));
        } else {
            $appendNode->parentNode->appendChild($clone);
        }
    }
     
}

/**
 * The form package make an API available for rendering a form and its elements
 * 
 * The form package currently does not fully support element arrays.
 * It can be done but it is not fully supported or tested.
 * 
 * @package Dom
 */
class Dom_Form 
{
    
    
    /**
     * @var DOMElement
     */
    protected $form = null;
    
    /**
     * An Array of Dom_FormElement objects
     * @var array
     */
    protected $elements = array();
    
    /**
     * @var Dom_Template
     */
    protected $parent = null;
    
    
    /**
     * __construct
     * 
     * @param DOMElement $form
     * @param array $element An array of form elements
     * @param Dom_Template $parent The parent object
     */
    function __construct($form, $elements, $parent) 
    {
        $this->form = $form;
        $this->parent = $parent;
        $this->elements = $elements;
    }
    
    /**
     * Set/unset the checkboxes and radioboxes.
     * <b>NOTE:</b> This is called by Dom_FormInputElement<br>
     *   $value is not required for checkboxes
     * 
     * @param string $name 
     * @param string $value 
     */
    function setCheckedByValue($name, $value = '') 
    {
        if (!isset($this->elements[$name])) {
            return;
        }
        $elements = $this->elements[$name];
        foreach($elements as $element) {
            if ($value != null && ($element->getAttribute('value') == $value)) {
                $element->setAttribute('checked', 'checked');
            }else{
                $element->removeAttribute('checked');
            }
        }
    }
    
    /**
     * Return the form element with the name.
     *
     * @param string $name
     * @param int $i optioinal index for multiple elements
     * @return Dom_FormElement 
     */
    function getFormElement($name, $i = 0) 
    {
        if (!$this->elementExists($name)) {
            return;
        }
        $element = $this->elements[$name][$i];
        $type = $element->nodeName;
        if ($type == 'input') {
            return new Dom_FormInput($element, $this);
        }elseif ($type == 'textarea') {
            return new Dom_FormTextarea($element, $this);
        }elseif ($type == 'select') {
            return new Dom_FormSelect($element, $this);
        }
    }
    
    /**
     * Get an array of form elements with the name value
     * Used for radioboxes and multi select lists
     *
     * @param string $name
     * @return array
     */
    function getFormElementList($name) 
    {
        if (!$this->elementExists($name)) {
            return array();
        }
        $nodeList = array();
        $n = count($this->elements[$name]);
        for($i=0; $i<$n; $i++) {
            $element = $this->elements[$name][$i];
            $type = $element->nodeName;
            if ($type == 'input') {
                $nodeList[] = new Dom_FormInput($element, $this);
            }elseif ($type == 'textarea') {
                $nodeList[] = new Dom_FormTextarea($element, $this);
            }elseif ($type == 'select') {
                $nodeList[] = new Dom_FormSelect($element, $this);
            }
        }
        return $nodeList;
    }
    
    /**
     * Return the number of elements in an element namespace
     * 
     * @param string $name
     * @return int 
     */
    function getNumFormElements($name) 
    {
        return count($this->elements[$name]);
    }
    
    /**
     * Get an array containing the form element names
     * 
     * @return array
     */
    function getElementNames() 
    {
        return array_keys($this->elements);
    }
    
    /**
     * Set a URL that defines where to send the data when 
     *  the submit button is pushed.
     * 
     * @param string $action
     */
    function setAction($value) 
    {
        if ($this->form != null) {
            $this->form->setAttribute('action', Dom_Template::objectToString($value));
        }
    }
    
    /**
     * The HTTP method for sending data to the action URL. 
     * Default is get.<br/>
     * Possible values are:<br/>
     * <ul>
     *   <li>'get'</li>
     *   <li>'post'</li>
     * </ul>
     * 
     * @param string $method 
     */
    function setMethod($value) 
    {
        if ($this->form != null) {
            $this->form->setAttribute('method', Dom_Template::objectToString($value));
        }
    }
    
    
    /**
     * Set the method used by the form.
     * Possible values are:<br/>
     * <ul>
     *   <li>'_blank'</li>
     *   <li>'_self' (default)</li>
     *   <li>'_parent'</li>
     *   <li>'_top'</li>
     * </ul>
     *
     * @param string $target
     */
    function setTarget($value) 
    {
        if ($this->form != null) {
            $this->form->setAttribute('target', Dom_Template::objectToString($value));
        }
    }
    
    /**
     * Append a hidden element to a form.
     * 
     * @param string $name
     * @param string $value
     */
    function appendHiddenElement($name, $value) 
    {
        if ($this->form != null) {
            $nl = $this->form->ownerDocument->createTextNode("\n");
            $node = $this->form->ownerDocument->createElement('input');
            $node->setAttribute('type', 'hidden');
            $node->setAttribute('name', $name);
            $node->setAttribute('value', Dom_Template::objectToString($value));
            $this->form->appendChild($node);
            $this->form->appendChild($nl);
        }
    }
    
    /**
     * Get an array of the hidden elements in this form
     *
     * @return unknown
     */
    function getHiddenElements()
    {
        $arr = array();
        foreach ($this->elements as $element) {
            $type = $element->nodeName;
            $inputType = $element->getAttribute('type');
            if ($type == 'input' && $inputType == 'hidden') {
                $arr[] = new Dom_FormInputElement($element, $this);
            }
        }
        return $arr;
    }
    
    /**
     * Get the form name.
     * 
     * @return string 
     * @deprecated This is nolonger valid XHTML. use getId()
     */
    function getName() 
    {
        if ($this->form != null) {
            return $this->form->getAttribute('name');
        }
    }
    
    function getId() 
    {
        if ($this->form != null) {
            return $this->form->getAttribute('id');
        }
    }
    
    /**
     * Get the DOMElement of this form object.
     *
     * @return DOMElement
     */
    function getNode()
    {
        return $this->form;
    }
    
    /**
     * Check if a repeat,choice,var,form (template property) exists.
     * @param string $property 
     * @param string $key 
     */
    private function elementExists($key) 
    {
        if (!array_key_exists($key, $this->elements)) {
            $e = new Sdk_Exception("Unknown form element 'name=\"$key\"' in document.");
            error_log($e->__toString());
            return false;
        }
        return true;
    }
    
    /**
     * Get the parent template for this form
     *
     * @return Dom_Template
     */
    function getTemplate() 
    {
        return $this->parent;
    }
}


/**
 * All form elements must use this class/interface.
 * 
 * @package Dom
 */
abstract class Dom_FormElement
{
    
    /**
     * This could be a single DOMElement or an array of DOMElement
     * @var DOMElement
     */
    protected $element = null;
    
    /**
     * @var Dom_Form
     */
    protected $form = null;
    
    
    /**
     * __construct
     * 
     * @param DOMElement $element
     * @param Dom_Form $form
     */
    function __construct($element, $form = null) 
    {
        $this->element = $element;
        $this->form = $form;
    }

    /**
     * Set the name of this element
     * 
     * @param string name
     */
    function setName($name) 
    {
        $this->element->setAttribute('name', $name);
    }
    
    /**
     * Get the name of this element
     * 
     * @return string The name of this element.
     */
    function getName() 
    {
        return $this->element->getAttribute('name');
    }
    
    /**
     * Get the DomElement node for this form element
     *
     * @return DOMElement
     */
    function getNode()
    {
        return $this->element;
    }
    
    /**
     * Get the parent DOM form object
     *
     * @return Dom_Form
     */
    function getForm()
    {
        return $this->form;
    }
    
    /**
     * Get the Element's Template
     *
     * @return Dom_Template
     */
    function getTemplate()
    {
        if ($this->form) {
            return $this->form->getTemplate();
        }
    }
    
    /**
     * Set the value of a form element.
     * 
     * Set value behaves different for different elements:
     *  o input => This is the element value attribute
     *  o checkbox/radio => The value to check/select
     *  o select => The value of the option to be selected
     *  o textarea => the content of the textarea
     *
     * @param string $value
     */
    abstract function setValue($value);
    
    /**
     * Return the value of the element, or the selected value.
     *
     * @return mixed A string or an array of strings for 
     *    multiple select elements
     */
    abstract function getValue();
    
    /**
     * Return the form element type attribute
     *
     * @return string
     */
    function getType() 
    {
        return $this->element->getAttribute('type');
    }
    
    /**
     * Disable this allement, add a disable attribute to the node
     */
    function disable() 
    {
        $attr = $this->element->setAttribute('disabled', 'disabled');
    }

    /**
     * get the disabled state of this node
     *
     * @return boolean
     */
    function isDisabled() 
    {
        return $this->element->hasAttribute('disabled');
        
    }
     
    /**
     * Set the attribute name and value
     * @param string name
     * @param string value
     */
    function setAttribute($name, $value) 
    {
        $this->element->setAttribute($name, $value);
    }
    
    /**
     * Set the name of this element
     * @return string
     */
    function getAttribute($name) 
    {
        return $this->element->getAttribute($name);    
    }
}


/**
 * A class that handle a forms input element.
 * 
 * @package Dom
 */
class Dom_FormInput extends Dom_FormElement 
{
    
    
    /**
     * Set the checked attribute of an element
     * 
     * @param boolean $b
     */
    function setChecked($b) 
    {
        if ($b) {
            $this->element->setAttribute('checked', 'checked');
        }else{
            $this->element->removeAttribute('checked');
        }
    }
    
    /**
     * Get the checked state of this element
     * 
     * @return boolean 
     */
    function isChecked() 
    {
        return $this->element->hasAttribute('checked');
    }
    
    /**
     * Set the value of this form element.
     * 
     * @param string $value
     */
    function setValue($value) 
    {
        if ($this->getType() == 'checkbox' || $this->getType() == 'radio') {
            $this->form->setCheckedByValue($this->getName(), Dom_Template::objectToString($value));
        }else{
            $this->element->setAttribute('value', Dom_Template::objectToString($value));
        }
    }
    
    /**
     * Return the value of theis form element
     * 
     * @return string
     */
    function getValue() 
    {
        return $this->element->getAttribute('value');
    }
    
}

/**
 * A class that handles a forms textarea element.
 * 
 * @package Dom
 */
class Dom_FormTextarea extends Dom_FormElement 
{
    
    
    /**
     * Set the value of this form element
     * 
     * @param string $value
     */
    function setValue($value) 
    {
      $dom = $this->element->ownerDocument;
      $textNode = $dom->createTextNode($value);
      $this->element->appendChild($textNode);
    }
    
    /**
     * Get the current text in the textarea
     * 
     * @return string
     */
    function getValue() 
    {
        return $this->element->nodeValue;
    }
}

/**
 * A class that handle a forms select element.
 * 
 * @package Dom
 */
class Dom_FormSelect extends Dom_FormElement 
{
    
    /**
     * set to false to add spaces (&#160; or &nbsp;)
     * @var boolean
     */
    private $useTextNode = true;
    
    
    /**
     * Use Text Nodes, Set to True by default
     *
     * @param boolean $b
     */
    function useTextNodes($b)
    {
        $this->useTextNode = $b;
    }
    
    /**
     * Append an 'Option' to this 'Select' object
     * 
     * If no value is supplied the text parameter is used as the value.
     * 
     * NOTE: Ensure no comment nodes are in the select's node tree.
     * @param string The text shown in the dropdown
     * @param string The value for the select option
     * @param string $optGroup Use this optgroup if it exists
     * @return DOMElement
     */
    function appendOption($text, $value = null, $optGroup = '') 
    {
        //vd($text, $value, $optGroup);
        $doc = $this->element->ownerDocument;
        $nl = $doc->createTextNode("\n");
        $option = $doc->createElement('option');
        if ($value === null) {
            $option->setAttribute('value', $text);
        } else {
            $option->setAttribute('value', Dom_Template::objectToString($value));
        }
        
        if ($this->useTextNode) {
            $text_el = $doc->createTextNode($text);
            $option->appendChild($text_el);
        } else {
            $text = ereg_replace('&( )', '&amp; ', $text);
            $option->nodeValue = $text;
        }
        
        $optGroupNode = null;
        if ($optGroup != null) {
            $optGroupNode = $this->findOptGroup($this->element, $optGroup);
        }
        if ($optGroupNode != null) {
            $optGroupNode->appendChild($nl);
            $optGroupNode->appendChild($option);
        } else {
            $this->element->appendChild($nl);
            $this->element->appendChild($option);
        }
        
        return $option;
    }
    
    /**
     * Append an 'OptGroup' to the base node or the optGroup 
     * 
     * 
     * @param string The label for the optGroup
     * @param string $optGroup Append to this optgroup if it exists
     * @return DOMElement
     */
    function appendOptGroup($label, $optGroup = '') 
    {
        $doc = $this->element->ownerDocument;
        $nl = $doc->createTextNode("\n");
        $option = $doc->createElement('optgroup');
        
        $option->setAttribute('label', $label);
        
        $optGroupNode = null;
        if ($optGroup != null) {
            $optGroupNode = $this->findOptGroup($this->element, $optGroup);
        }
        if ($optGroupNode != null) {
            $optGroupNode->appendChild($nl);
            $optGroupNode->appendChild($option);
        } else {
            $this->element->appendChild($nl);
            $this->element->appendChild($option);
        }
        return $option;
    }
    
    /**
     * Set the selected value of the form element
     * 
     * @param mixed $value A string for single, an array for multiple
     */
    function setValue($value) 
    {
          if (is_array($value)) {
              if ($this->isMultiple()) {
                  foreach ($value as $k => $v) {
                      $option = $this->findOption($this->element, $v);
                      if ($option != null) {
                          $option->setAttribute('selected', 'selected');
                      }
                  }
              } else {
                  $option = $this->findOption($this->element, $value[0]);
                  if ($option != null) {
                      $option->setAttribute('selected', 'selected');
                  }
              }
          } else {
              if (!$this->isMultiple()) {
                  $this->clearSelected();
              }
              $option = $this->findOption($this->element, $value);
              if ($option != null) {
                  $option->setAttribute('selected', 'selected');
              }
          }
    }
    
    /**
     * Return the selected value, 
     * Will return an array if  multiple select is enabled. 
     * 
     * @return mixed Returns null if nothing selected.
     */
    function getValue() 
    {
        $selected = $this->findSelected($this->element);
        if (count($selected) > 0) {
            if ($this->isMultiple()) {
                return $selected;
            }else{
                return $selected[0];
            }
        }
    }
    
    
    
    /**
     * Clear this 'select' element of all its 'option' elements.
     * 
     */
    function removeOptions() 
    {
        while ($this->element != null && $this->element->hasChildNodes()) {
            $this->element->removeChild($this->element->childNodes->item(0));
        }
    }
    
    /**
     * Clear all selected elements
     * 
     */
     function clearSelected() 
     {
        $this->clearSelectedFunction($this->element);
     }
     
     /**
      * Find the opt group node with the name
      *
      * @param DOMElement $node
      * @param string $name
      * @return DOMElement
      */
    private function clearSelectedFunction($node)
    {
        if ($node->nodeType == XML_ELEMENT_NODE) {
            if ($node->nodeName == 'option' && $node->hasAttribute('selected')) {
                $node->removeAttribute('selected');
            }
            
            $childNodes = $node->childNodes;
            foreach($node->childNodes as $child) {
                $this->clearSelectedFunction($child);
            }
        }
    }
     
    /**
     * Find the opt group node with the name
     *
     * @param DOMElement $node
     * @param string $name
     * @return DOMElement
     */
    function findOptGroup($node, $name)
    {
        $foundNode = null;
        if ($node->nodeType == XML_ELEMENT_NODE) {
            if ($node->nodeName == 'optgroup' && $node->getAttribute('label') == $name) {
                return $node;
            }
            
            $childNodes = $node->childNodes;
            foreach($node->childNodes as $child) {
                $fNode = $this->findOptGroup($child, $name);
                if ($fNode != null) {
                    $foundNode = $fNode;
                }
            }
        }
        return $foundNode;
    }
     
    /**
     * Find an option node
     *
     * @param DOMElement $node
     * @param string $value
     * @return DOMElement
     */
    function findOption($node, $value)
    {
        $foundNode = null;
        if ($node->nodeType == XML_ELEMENT_NODE) {
            if ($node->nodeName == 'option' && $node->getAttribute('value') == $value) {
                return $node;
            }
            
            $childNodes = $node->childNodes;
            foreach($node->childNodes as $child) {
                $fNode = $this->findOption($child, $value);
                if ($fNode != null) {
                    $foundNode = $fNode;
                }
            }
        }
        return $foundNode;
    }
     
    /**
     * Find the selected values to this select box
     *
     * @param DOMElement $node
     * @return array
     */
    function findSelected($node)
    {
        $foundNodes = array();
        if ($node->nodeType == XML_ELEMENT_NODE) {
            if ($node->nodeName == 'option' && $child->hasAttribute('selected')) {
                return $node;
            }
            
            $childNodes = $node->childNodes;
            foreach($node->childNodes as $child) {
                $fNode = $this->findOption($child, $value);
                if ($fNode != null) {
                    $foundNodes[] = $fNode;
                }
            }
        }
        return $foundNodes;
    }
    
    /**
     * Check if the opt group exists
     *
     * @param string $name
     * @return boolean
     */
    function optGroupExists($name)
    {
        return $this->findOptGroup($this->element, $name) != null;
    }
    
    /**
     * Set the select list to handle multiple selections
     * <b>NOTE:</b> When multiple is dissabled and multiple elements are selected 
     *  it behaviour is unknown and browser specific.
     * 
     * @param boolean $b 
     */
    function enableMultiple($b) 
    {
        if ($b) {
            $this->element->setAttribute('multiple', 'multiple');
        }else{
            $this->element->removeAttribute('multiple');
        }
    }
    
    /**
     * Return if this is a multiple select or not.
     * 
     * @return boolean Returns true if muliple selects are allowed
     */
    function isMultiple() 
    {
        return $this->element->hasAttribute('multiple');
    }
}



?>