<?
	/**
	* @package	PDFClass
	* @author		Steven Surowiec
	* @version CVS: $Id: pdf_core.class.php,v 1.175 2007/10/02 01:22:56 cvs Exp $
	* @todo Add support for cellspacing and cellpadding to table drawing
	* @todo Investigate and fix gzip so that it does not crash PDFs
	* @todo Add hotlink support, external and internal
	* @todo Add bookmark support
	* @todo Add table of contents support
	* @todo Test, fix and finalize Jpeg support
	* @todo Add PNG and additional image types support
	*/
	
	/**
	 * Define some constants that we can use
	 */
	define('PDF_DEFAULT_FORMAT', FALSE);

	
	/**
	 * This class handles our PDF generation and management.
	 * @package PDFClass
	 */
	class PDFClass {		
		/**
		 * This holds current exception text.
		 * @var string
		 * @access private
		 */
		private $exception = '';
		
		/**
		 * This holds current exception error number.
		 * @var int
		 * @access private
		 */
		private $errno = 0;
		
		/**
		 * This holds the current font family that is in use.
		 * @var string
		 * @access private
		 */
		private $_font_family = '';
		
		/**
		 * This holds the current font style that is in use.
		 * @var string
		 * @access private
		 */
		private $_font_style = '';
		
		/**
		 * This holds the font resource information of the current font in use.
		 * @var array
		 * @access private
		 */
		private $_current_font;	
		
		/**
		 * This holds the current font size that is in use, in points.
		 * @var int
		 * @access private
		 */
		private $_font_size = 12;
		
		/**
		 * This holds the spacing of characters to use when writing text
		 * @var int
		 * @access private
		 */
		private $_char_spacing = 0;
		
		/**
		 * This holds the spacing of words to use when writing text
		 * @var int
		 * @access private
		 */
		private $_word_spacing = 0;
		
		/**
		 * This holds our current text deviation so we can easily switch between text methods 
		 * without having to carry it from signature to signature.
		 * @var int
		 * @access private
		 */
		private $_text_deviation = 0;
		
		/**
		 * List of files currently open. Contains all information for files such as resource id,
		 * buffer, images, fonts, shapes, colors, etc.
		 * @var array
		 * @access private
		 */
		private $files = Array();
		
		/**
		 * This holds our data set for the current font should one exist.
		 * @var array
		 * @access private
		 */
		private $_font_data = Array();
		
		/**
		 * This is a list of core fonts. These fonts are natively supported by Acrobat Reader so no additional
		 * support is necessary to support them.
		 * @var array
		 * @access private
		 */
		private $_core_fonts = Array();
		
		/**
		 * This holds data about our current draw color that is set
		 * @var array
		 * @access private
		 */
		private $_draw_color = Array();
		
		/**
		 * This holds data about our current fill color that is set
		 * @var array
		 * @access private
		 */
		private $_fill_color = Array();
		
		/**
		 * This holds the page number of the last modified page should the page pointer be changed
		 * @var int
		 * @access private
		 */
		private $_pre_page = false;
		
		/**
		 * This holds all available error codes that may be returned due to exceptions
		 * @var array
		 * @access private
		 */
		private $_error_codes = Array(
			999 => 'Unknown error',
			1000 => 'PDFClass document with name already in use',
			1001 => 'Invalid size format provided',
			1002 => 'Invalid orientation provided',
			1003 => 'Unable to create new PDFClass file: unknown error',
			1004 => 'Invalid PDFClass resource ID',
			1005 => 'Unknown PDFClass resource ID',
			1006 => 'Cannot modify headers, headers already sent',
			1007 => 'Requested page not within scope of document',
			1008 => 'File has no, or invalid, type set',
			1009 => 'Unsupported file type',
			1010 => 'Wordspacing setting must be an integer',
			1011 => 'Unknown PDFClass table ID',
			1012 => 'Unable to write table with empty data set',
			1013 => 'Unable to load font file',
		);
		
		/**
		 * This holds all available page sizes, it is public so that it can be manually added to
		 * @var array
		 * @access public
		 */
		public $file_sizes = Array(
			'4a0' => Array('4767.87', '6740.79'),
			'2a0' => Array('3370.39', '4767.87'),
			'a0' => Array('2383.94', '3370.39'),
			'a1' => Array('1683.78', '2383.94'),
			'a2' => Array('1190.55', '1683.78'),
			'a3' => Array('841.89', '1190.55'),
			'a4' => Array('595.28', '841.89'),
			'a5' => Array('420.94', '595.28'),
			'a6' => Array('297.64', '420.94'),
			'a7' => Array('209.76', '297.64'),
			'a8' => Array('147.40', '209.76'),
			'a9' => Array('104.88', '147.40'),
			'a10' => Array('73.70', '104.88'),
			'b0' => Array('2834.64', '4008.18'),
			'b1' => Array('2004.09', '2834.64'),
			'b2' => Array('1417.32', '2004.09'),
			'b3' => Array('1000.62', '1417.32'),
			'b4' => Array('708.66', '1000.62'),
			'b5' => Array('498.89', '708.66'),
			'b6' => Array('354.33', '498.89'),
			'b7' => Array('249.44', '354.33'),
			'b8' => Array('175.74', '294.44'),
			'b9' => Array('124.72', '175.74'),
			'b10' => Array('87.87', '124.72'),
			'c0' => Array('2599.37', '3676.53'),
			'c1' => Array('1836.85', '2599.37'),
			'c2' => Array('1298.26', '1836.85'),
			'c3' => Array('918.42', '1298.26'),
			'c4' => Array('649.13', '918.42'),
			'c5' => Array('459.21', '649.13'),
			'c6' => Array('323.14', '459.21'),
			'c7' => Array('229.60', '323.14'),
			'c8' => Array('161.57', '229.60'),
			'c9' => Array('113.38', '161.57'),
			'c10' => Array('79.37', '113.38'),
			'letter' => Array('612', '792'),
			'legal' => Array('612', '1008'),
			'executive' => Array('521.86', '756'),
			'folio' => Array('612', '936'),
			'archa' => Array('864.56', '649.13'),
			'archb' => Array('1295.43', '864.56'),
			'archc' => Array('1729.13', '1295.43'),
			'archd' => Array('2590.86', '1729.13'),
			'arche' => Array('3455.43', '2590.86'),
			'arche1' => Array('3024.56', '2160'),
		);
		
		/**
		 * This is our constructor. This defines our list of core fonts for us.
		 * @return void
		 * @access public
		 */	
		public function PDFClass() {
			// define our core fonts
			$this->_core_fonts = Array('courier' => 'Courier', 'courierB' => 'Courier-Bold', 'courierI' => 'Courier-Oblique', 'courierBI' => 'Courier-BoldOblique', 'helvetica' => 'Helvetica', 
				'helveticaB' => 'Helvetica-Bold',  'helveticaI' => 'Helvetica-Oblique', 'helveticaBI' => 'Helvetica-BoldOblique', 'times' => 'Times-Roman', 'timesB' => 'Times-Bold', 
				'timesI' => 'Times-Italic', 'timesBI' => 'Times-BoldItalic', 'symbol' => 'Symbol', 'zapfdingbats' => 'ZapfDingbats');
		}
		
		/**
		 * This method allows us to create a new file within our class.
		 * Sets default values for pdfs and starts the buffer storage.
		 * Supported orientations are P, for portrait, and L, for landscape.
		 * Supported forms are; 4A0, 2A0, A3, A4, A5, Letter, Legal.
		 * The file name may be changed later on by use of {@link setVar}.
		 * @param string $name is the name of the file to use
		 * @param string $orientation is the orientation of the document
		 * @param mixed $format is the page format of the document or an array with two indexes, matching the width and height of the pages respectively.
		 * @return int
		 * @access public
		 */
		public function createNewFile($name, $orientation = 'P', $format = NULL) {
			// make sure the extension was specified, if not add it ourselves
			$ext = end(explode('.', $name));
			if(strtolower($end) != 'pdf') $name .= '.pdf';
			
			// make sure this file name doesn't already exist
			if(!$this->checkFileName($name)) {
				$file = Array('name' => $name, 'buffer' => "%PDF-1.6\n");
			} else {
				$this->setException(1000);
				throw new Exception($this->exception);
				return FALSE;
			}
			
			// are we using a default format?
			if(is_null($format)) {
				if(PDF_DEFAULT_FORMAT != FALSE) {
					$format = PDF_DEFAULT_FORMAT;
				} else {
					$format = 'letter';
				}
			}

			// determine page size based on format
			if(is_array($format)) {
				/* make sure both indexes are numeric */
				if(!is_numeric($format[0]) || !is_numeric($format[1])) {
					$this->setException(1001);
					throw new Exception($this->exception);
					return FALSE;
				} 
			} else {
				$format = strtolower($format);
				if($this->file_sizes[$format]) {
					$format = $this->file_sizes[$format];
				} else {
					$this->setException(1001);
					throw new Exception($this->exception);
					return FALSE;
				}
			}
			$file['w'] = $format[0];
			$file['h'] = $format[1];
			
			// set page orientation
			$orientation = strtolower($orientation);
			if ($orientation == 'l' || $orientation == 'landscape') {
				$w = $file['w'];
				$file['w'] = $file['h'];
				$file['h'] = $w;
			} else if ($orientation != 'p' && $orientation != 'portrait') {
				$this->setException(1002);
				throw new Exception($this->exception);
				return FALSE;
			}
			
			// set this to false because it was breaking stuff
			//$file['compress'] = function_exists('gzcompress')?true:false;
			$file['compress'] = false;
			
			// set some defaults for this file
			$file['page'] = 0;			// current page number
			$file['pages'] = Array();	// array of pages in document
			$file['state'] = 1;			// document state (0 = initialised, 1 = opened but no page opened, 2 = page opened, 3 = document closed)
			$file['n'] = 2;				// current object number
			$file['offsets'] = Array();	// array of object offsets
			$file['fonts'] = Array();	// array of used fonts
			$file['vars'] = Array();	// array used to hold variable data
			$file['fill_color'] = '';
			$file['draw_color'] = '';
			$file['line_width'] = '';
			$file['images'] = Array();
			$file['tables'] = Array();	// array of data used for tables
			$file['bookmarks'] = Array();	// array of bookmark data
			
			// add this file to our list, and return it's id
			$this->files[] = $file;
			$resource = $this->getResourceId($name);
			if($resource === FALSE) {
				$this->setException(1003);
				throw new Exception($this->exception);
				return FALSE;
			}
			return $resource;
		}
		
		/**
		 * This method closes our file. Closing the file finalizes the buffer for output.
		 * @param int $resource is the resource id of the document to close
		 * @return void
		 * @access public
		 */		
		public function closeFile($resource) {
			// make sure our resource is valid
			if($this->checkResource($resource) === FALSE) {
				throw new Exception($this->exception);
				return FALSE;
			}
			
			// if we dont have a page, add one to make this a valid pdf file
			if($this->files[$resource]['page'] == '0') {
				$this->addPage($resource);
			}
			
			// set state to page closed
			$this->files[$resource]['state'] = 1;
			
			// output page and resource buffers
			$this->putPages($resource);
			$this->putResources($resource);
			
			// print some document information
			$this->setVar($resource, 'Producer', 'My PDF Class');
			$this->setVar($resource, 'CreationDate', sprintf('D:%s', date('YmdHis')));
			$this->putVars($resource);
			
			// print catalog
			$this->newobj($resource);
			$this->out($resource, '<<');
			$this->out($resource, '/Type /Catalog');
			$this->out($resource, '/Pages 1 0 R');
			$this->out($resource, '/OpenAction [3 0 R /FitH null]');
			$this->out($resource, '/PageLayout /OneColumn');
			$this->out($resource, '>>');
			$this->out($resource, 'endobj');
			
			// print cross reference
			$start_xref = strlen($this->files[$resource]['buffer']);
			$this->out($resource, 'xref');
			$this->out($resource, '0 ' . ($this->files[$resource]['n'] + 1));
			$this->out($resource, '0000000000 65535 f ');
			
			// loop through all objects and output their offset
			for ($i = 1; $i < $this->files[$resource]['n']; $i++) {
				$this->out($resource, sprintf('%010d 00000 n ', $this->_offsets[$i]));
			}
			
			// print trailer
			$this->out($resource, 'trailer');
			$this->out($resource, '<<');
			
			// the total number of objects
			$this->out($resource, '/Size ' . ($this->files[$resource]['n'] + 1));
			
			// the root object
			$this->out($resource, '/Root ' . $this->files[$resource]['n'] . ' 0 R');
			
			// the document information object
			$this->out($resource, '/Info '. ($this->files[$resource]['n'] + 1) .' 0 R');
			$this->out($resource, '>>');
			$this->out($resource, 'startxref');
			$this->out($resource, $start_xref);
			$this->out($resource, '%%EOF');
			
			// set document state to closed
			$this->files[$resource]['state'] = 3;
		}
		
		/**
		 * This method outputs our document to the browser. If var $download is set to <i>true</i> then the 
		 * document will be presented as a downloadable.
		 * @param int $resource is the resource id of the document to output
		 * @return void
		 * @access public
		 */				
		public function outputFile($resource) {
			// make sure our resource is valid
			if($this->checkResource($resource) === FALSE) {
				throw new Exception($this->exception);
				return FALSE;
			}
			
			// if document is not yet closed, close it now
			if($this->files[$resource]['state'] < 3) {
				$this->closeFile($resource);
			}
			
			// make sure no content is already sent
			if(headers_sent()) {
				$this->setException('Cannot modify headers, headers already sent', 1006);
				throw new Exception($this->exception);
				return FALSE;
			}
			
			// if we have a name variable, see if it has an extension
			if($this->files[$resource]['vars']['name']) {
				$ext = end(explode('.', $this->files[$resource]['vars']['name']));
				if(strtolower($ext) == 'pdf') {
					$name = $this->files[$resource]['vars']['name'];
				} else {
					$name = $this->files[$resource]['vars']['name'].'.pdf';
				}
			} else {
				$name = $this->files[$resource]['name'];
			}
						
			// do we want to offer it for download, or display it inline?
			if($this->files[$resource]['vars']['download']) {
				// offer file for download and do some browser checks for correct download
				$agent = trim($_SERVER['HTTP_USER_AGENT']);
				if((preg_match('|MSIE ([0-9.]+)|', $agent, $version)) || (preg_match('|Internet Explorer/([0-9.]+)|', $agent, $version))) {
					Header('Content-Type: application/x-msdownload');
					Header('Content-Length: ' . strlen($this->files[$resource]['buffer']));
					if($version == '5.5') {
						Header('Content-Disposition: filename="' . $name . '"');
					} else {
						Header('Content-Disposition: attachment; filename="' . $name . '"');
					}
				} else {
					Header('Content-Type: application/pdf');
					Header('Content-Length: ' . strlen($this->files[$resource]['buffer']));
					Header('Content-disposition: attachment; filename=' . $name);
				}
			} else if(!$this->files[$resource]['vars']['display']) {
				$len = strlen($this->files[$resource]['buffer']);
				Header("Content-type: application/pdf");
				Header("Content-Length: $len");
				Header("Content-Disposition: inline; filename=". $name);
			}
			
			echo $this->files[$resource]['buffer'];
		}
		
		/**
		 * This method sets the current page to be used until another page 
		 * is added or the pointer is again changed. Returns <i>false</i> if invalid
		 * page is provided. Returns <i>true</i> otherwise. 
		 * @param int $resource is the resource id of the document to use
		 * @param int $page is the page number to change the pointer to
		 * @return boolean
		 * @access public
		 */
		public function setPagePointer($resource, $page) {
			// make sure our resource is valid
			if($this->checkResource($resource) === FALSE) {
				throw new Exception($this->exception);
				return FALSE;
			}
			
			// make sure our page is a number
			if(!is_numeric($page) || $page < 1) return FALSE;
			
			// make sure its within the contraints of our number of pages
			if($page > (count($this->files[$resource]['pages']) - 1)) {
				$this->setException(1007);
				throw new Exception($this->exception);
				return FALSE;
			}
			
			// save the current page for future reference
			$this->_prev_page = $this->files[$resource]['page'];
			
			// seems good, set our pointer and return true
			$this->files[$resource]['page'] = $page;
			return true;
		}
		
		/**
		 * This method gets our current page pointer
		 * @param int $resource is the resoruce id of the document to use
		 * @return int
		 * @access public
		 */
		public function getPagePointer($resource) {
			// make sure our resource is valid
			if($this->checkResource($resource) === FALSE) {
				throw new Exception($this->exception);
				return FALSE;
			}
			
			// return our page pointer
			return $this->files[$resource]['page'];
		}
		
		/**
		 * This method resets the page pointer if it is changed with {@link setPagePointer}
		 * @param int $resource is the resource id of the document to use
		 * @return boolean
		 * @access public
		 */
		public function resetPagePointer($resource) {
			// make sure our resource is valid
			if($this->checkResource($resource) === FALSE) {
				throw new Exception($this->exception);
				return FALSE;
			}
			
			// reset our page pointer and return true
			if($this->_prev_page) {
				$this->files[$resource]['page'] = $this->_prev_page;
			}
			return true;
		}
		
		/**
		 * This method returns the id of the last page of the document
		 * @param int $resource is the resource id of the document to use
		 * @return int
		 * @access public
		 */		
		public function getLastPage($resource) {
			// make sure our resource is valid
			if($this->checkResource($resource) === FALSE) {
				throw new Exception($this->exception);
				return FALSE;
			}
			
			// return our last page number
			return (count($this->files[$resource]['pages'])	- 1);
		}
		
		/**
		 * This method sets our fill color for use with fonts and fills
		 * @param int $resource is the resource id of the document to use
		 * @param string $cs is the color style to use
		 * @param int $c1 is the first color to use
		 * @param int $c2 is the second color to use
		 * @param int $c3 is the third color to use
		 * @param int $c4 is the fourth color to use
		 * @return void
		 * @access public
		 */		
		public function setFillColor($resource, $cs = 'rgb', $c1, $c2 = 0, $c3 = 0, $c4 = 0) {
			// make sure our resource is valid
			if($this->checkResource($resource) === FALSE) {
				throw new Exception($this->exception);
				return FALSE;
			}
			
			// figure out what type of coloring we're using
			$cs = strtolower($cs);
			switch($cs) {
				case 'rgb':
					// three component rgb color
					$this->files[$resource]['fill_color'] = sprintf('%.3f %.3f %.3f rg', $c1, $c2, $c3);
					break;
				case 'cmyk':
					// four component cmyk color
					$this->files[$resource]['fill_color'] = sprintf('%.3f %.3f %.3f %.3f k', $c1, $c2, $c3, $c4);
					break;
				default:
					// single component gray scale color
					$this->files[$resource]['fill_color'] = sprintf('%.3f g', $c1);
					break;
			}
			
			// if the document has been started, write this to the buffer
			if($this->files[$resource]['page'] > 0) {
				$this->out($resource, $this->files[$resource]['fill_color']);
			}
			
			// store this data incase we need it later
			$this->_fill_color = Array(
				'cs' => $cs,
				'cs1' => $cs1,
				'cs2' => $cs2,
				'cs3' => $cs3,
				'cs4' => $cs4
			);
		}
		
		/**
		 * This method sets our draw color for use with strokes
		 * @param int $resource is the resource id of the document to use
		 * @param string $cs is the color style to use
		 * @param int $c1 is the first color to use
		 * @param int $c2 is the second color to use
		 * @param int $c3 is the third color to use
		 * @param int $c4 is the fourth color to ues
		 * @return void
		 * @access public
		 */	
		public function setDrawColor($resource, $cs = 'rgb', $c1, $c2 = 0, $c3 = 0, $c4 = 0) {
			// make sure our resource is valid
			if($this->checkResource($resource) === FALSE) {
				throw new Exception($this->exception);
				return FALSE;
			}
			
			// figure out what type of coloring we're using
			$cs = strtolower($cs);
			switch($cs) {
				case 'rgb':
					// three component rgb color
					$this->files[$resource]['draw_color'] = sprintf('%.3f %.3f %.3f RG', $c1, $c2, $c3);
					break;
				case 'cmyk':
					// four component cmyk color
					$this->files[$resource]['draw_color'] = sprintf('%.3f %.3f %.3f %.3f K', $c1, $c2, $c3, $c4);
					break;
				default:
					// single component gray scale color
					$this->files[$resource]['draw_color'] = sprintf('%.3f G', $c1);
					break;
			}
			
			// if the document has been started, write this to the buffer
			if($this->files[$resource]['page'] > 0) {
				$this->out($resource, $this->files[$resource]['draw_color']);
			}
			
			// store this data incase we need it later
			$this->_draw_folor = Array(
				'cs' => $cs,
				'cs1' => $cs1,
				'cs2' => $cs2,
				'cs3' => $cs3,
				'cs4' => $cs4
			);
		}

		/**
		 * This method draws a line.
		 * @param int $resource is the resource id of the document to use
		 * @param int $x1 is the x-coordinate of the starting position
		 * @param int $y1 is the y-coordinate of the starting position
		 * @param int $x2 is the x-coordinate of the ending position
		 * @param int $y2 is the y-coordinate of the ending position
		 * @return void
		 * @access public
		 */			
		public function drawLine($resource, $x1, $y1, $x2, $y2) {
			// make sure our resource is valid
			if($this->checkResource($resource) === FALSE) {
				throw new Exception($this->exception);
				return FALSE;
			}
			
			// draw line
			$this->out($resource, sprintf('%.2f %.2f m %.2f %.2f l S', $x1, $this->files[$resource]['h'] - $y1, $x2, $this->files[$resource]['h'] - $y2));			
		}
		
		/**
		 * This method draws a rectangle
		 * @param int $resource is the resource id of the document to use
		 * @param int $x is the x-coordinate of the starting position
		 * @param int $y is the y-coordinate of the starting position
		 * @param int $width is the width of the rectangle
		 * @param int $height is the height of the rectangle
		 * @param string $style is the fill style of the rectangle
		 * @return void
		 * @access public
		 */
		public function drawRectangle($resource, $x, $y, $width, $height, $style = '') {
			// make sure our resource is valid
			if($this->checkResource($resource) === FALSE) {
				throw new Exception($this->exception);
				return FALSE;
			}
			
			// determine style
			$style = strtolower($style);
			switch($style) {
				case 'f':	// fill only
					$op = 'f';
					break;
				case 'fd':	// draw and fill
				case 'df':
					$op = 'B';
					break;
				default:	// draw only
					$op = 'S';
					break;
			}
			
			// write this to the buffer
			$this->out($resource, sprintf('%.2f %.2f %.2f %.2f re %s', $x, $this->files[$resource]['h'] - $y, $width, -$height, $op));
		}
		
		/**
		 * This method draws a circle
		 * @param int $resource is the resource id of the document to use
		 * @param int $x is the x-coordinate of the starting position
		 * @param int $y is the y-coordinate of the starting position
		 * @param int $r is the radius of the circle
		 * @param string $style is the fill style of the rectangle
		 * @return void
		 * @access public
		 */
		public function drawCircle($resource, $x, $y, $r, $style = '') {
			// make sure our resource is valid
			if($this->checkResource($resource) === FALSE) {
				throw new Exception($this->exception);
				return FALSE;
			}
			
			// determine style
			$style = strtolower($style);
			switch($style) {
				case 'f':	// fill only
					$op = 'f';
					break;
				case 'fd':	// draw and fill
				case 'df':
					$op = 'B';
					break;
				default:	// draw only
					$op = 'S';
					break;
			}
			
			$y = $this->files[$resource]['h'] - $y;		// adjust y to compensate for top-to-bottom
			$b = $r * 0.552;		// length of the Bezier controls
			
			/* move from the given origin, and set the current point to the start of the first Bezier curve */
			$c = sprintf('%.2f %.2f m', $x - $r, $y);
			$x = $x - $r;
			
			// first circle quarter
			$c .= sprintf(' %.2f %.2f %.2f %.2f %.2f %.2f c', $x, ($y + $b), ($x + $r - $b), ($y + $r), ($x + $r), ($y + $r));
			
			// set x/y to the final point
			$x = $x + $r;
			$y = $y + $r;
			
			// second circle quater
			$c .= sprintf(' %.2f %.2f %.2f %.2f %.2f %.2f c', ($x + $b), $y, ($x + $r), ($y - $r + $b), ($x + $r), ($y - $r));
			
			// set x/y to the final point
			$x = $x + $r;
			$y = $y - $r;
			
			// third circle quarter
			$c .= sprintf(' %.2f %.2f %.2f %.2f %.2f %.2f c', $x, ($y - $b), ($x - $r + $b), ($y - $r), ($x - $r), ($y - $r));
			
			// set x/y to the final point
			$x = $x - $r;
			$y = $y - $r;
			
			// fourth circle quater
			$c .= sprintf(' %.2f %.2f %.2f %.2f %.2f %.2f c %s', ($x - $b), $y, ($x - $r), ($y + $r - $b), ($x - $r), ($y + $r), $op);
			
			// write this to buffer
			$this->out($resource, $c);
		}
		
		/**
		 * This method sets the width to use when drawing a line
		 * @param int $resource is the resource id of the document to apply this to.
		 * @param int $width is the width to use	
		 * @return void
		 * @access public	 
		 */
		public function setLineWidth($resource, $width) {
			// make sure our resource is valid
			if($this->checkResource($resource) === FALSE) {
				throw new Exception($this->exception);
				return FALSE;
			}
			
			// set width and output to buffer if needed
			$this->files[$resource]['line_width'] = $width;
			if($this->files[$resource]['page'] > 0) {
				$this->out($resource, sprintf('%.2f w', $width));
			}
		}
		
		/**
		 * This method gets and returns our current line with
		 * @param int $resource is the resource id of the document to use
		 * @return int
		 * @access public
		 */
		public function getLineWidth($resource) {
			// make sure our resource is valid
			if($this->checkResource($resource) === FALSE) {
				throw new Exception($this->exception);
				return FALSE;
			}
			
			// return our line width
			return $this->files[$resource]['line_width'];
		}
		
		/**
		 * Sets a document variable to the provided value.
		 * @param int $resource is the resource id of the document to apply this to.
		 * @param string $name is the name of the variable to set.
		 * @param string $value is the value of the variable.
		 * @return void
		 * @access public
		 */
		public function setVar($resource, $name, $value) {
			// make sure our resource is valid
			if($this->checkResource($resource) === FALSE) {
				throw new Exception($this->exception);
				return FALSE;
			}
			
			// set this variable
			$this->files[$resource]['vars'][$name] = $value;			
		}
		
		/**
		 * Retrieves a document variable.
		 * @param int $resource is the resouce id of the document to look in.
		 * @param string $name is the name of the variable to retrieve.
		 * @return mixed
		 * @access public
		 */
		public function getVar($resource, $name) {
			// make sure our resource is valid
			if($this->checkResource($resource) === FALSE) {
				throw new Exception($this->exception);
				return FALSE;
			}
			
			// get this variable
			return $this->files[$resource]['vars'][$name];
		}
		
		/**
		 * Returns an array of declared document variables for a provided resource.
		 * @param int $resource is the resource id of the document to look in.
		 * @return array
		 * @access public
		 */
		public function getDeclaredVars($resource) {
			// make sure our resource is valid
			if($this->checkResource($resource) === FALSE) {
				throw new Exception($this->exception);
				return FALSE;
			}
			
			// return the variable set
			return $this->files[$resource]['vars'];
		}
		
		/**
		 * This method adds a page to the end of the document.
		 * @param int $resource is the resource id of the document to use
		 * @param boolean $skip_num 
		 * @return void
		 * @access public
		 */
		public function addPage($resource, $skip_num = FALSE) {
			// make sure our resource is valid
			if($this->checkResource($resource) === FALSE) {
				throw new Exception($this->exception);
				return FALSE;
			}
			
			// increment our page counter
			$this->files[$resource]['page']++;
			
			// default page buffer
			$this->files[$resource]['pages'][$this->files[$resource]['page']] = '';
			
			// set state of document to page opened
			$this->files[$resource]['state'] = 2;
			
			// check if a font has been set before this page
			if($this->_font_family) {
				$this->setFont($resource, $this->_font_family, $this->_font_style, $this->_font_size);
			}
			
			// check if fill_color has been set before this page
			if($this->files[$resource]['fill_color'] != '0 g') {
				$this->out($resource, $this->files[$resource]['fill_color']);
			}
			
			// check if draw color has been set before this page
			if($this->files[$resource]['draw_color'] != '0 G') {
				$this->out($resource, $this->files[$resource]['draw_color']);
			}
			
			// check if line width ahs been set before this page
			if($this->files[$resource]['line_width'] != 1) {
				$this->out($resource, $this->files[$resource]['line_width']);
			}
			
			// are we displaying page numbers for this document?
			if($this->files[$resource]['vars']['page_numbers'] && !$skip_num) {
				// store the existing font settings
				$font_family = $this->_font_family;
				$font_style = $this->_font_style;
				$font_size = $this->_font_size;
				
				// get our font settings
				if(!$this->files[$resource]['vars']['page_numbers_font']) $this->files[$resource]['vars']['page_numbers_font'] = 'Courier';
				if(!$this->files[$resource]['vars']['page_numbers_size']) $this->files[$resource]['vars']['page_numbers_size'] = 8;
				if(!$this->files[$resource]['vars']['page_numbers_x']) $this->files[$resource]['vars']['page_numbers_x'] = 20;
				if(!$this->files[$resource]['vars']['page_numbers_y']) $this->files[$resource]['vars']['page_numbers_y'] = 20;
				
				// assign our font for the page number
				$this->setFont($resource, $this->files[$resource]['vars']['page_numbers_font'], $this->files[$resource]['vars']['page_numbers_style'], $this->files[$resource]['vars']['page_numbers_size']);
				
				// add the page number
				$this->addText($resource, $this->files[$resource]['vars']['page_numbers_x'], $this->files[$resource]['vars']['page_numbers_y'], $this->files[$resource]['page']);
				
				// re-assign font to previous setting
				$this->setFont($resource, $font_family, $font_style, $font_size);
			}
		}
		
		/**
		 * Allows the adding of an image to the document.
		 * @param int $resource is the resource of the document to use
		 * @param string $file is the path to the file on the server
		 * @param int $x is the x-coordinate of the starting position
		 * @param int $y is the y-coordinate of the starting position
		 * @param int $width is the width of the image
		 * @param int $height is the height of the image
		 * @return void
		 * @access public
		 */
		public function addImage($resource, $file, $x, $y, $width = 0, $height = 0) {
			// make sure our resource is valid
			if($this->checkResource($resource) === FALSE) {
				throw new Exception($this->exception);
				return FALSE;
			}
			
			// make sure we have this image
			if(!isset($this->files[$resource]['images'][$file])) {
				// get our extension
				if(($pos = substr($file, '.')) === false) {
					$this->setException(1008);
					throw new Exception($this->exception);
					return FALSE;
				}			
				$type = strtolower(substr($file, $pos+1));
				
				// make sure we have a jpeg
				if($type == 'jpg' || $type = 'jpeg') {
					$info = $this->parseJPEG($file);
				} else {
					$this->setException(1009);
					throw new Exception($this->exception);
					return FALSE;
				}
				
				// set the image object id
				$info['i'] = count($this->files[$resource]['images']) + 1;
				
				// set image to array
				$this->files[$resource]['images'][$file] = $info;
			} else {
				$info = $this->files[$resource]['images'][$file];
			}
			
			// do auto height/width calculations if we have to
			if(empty($width) && empty($height)) {
				$width = $info['w'];
				$height = $info['h'];
			} else if(empty($width)) {
				$width = $height * $info['w'] / $info['h'];
			} else if(empty($height)) {
				$height = $width * $info['h'] / $info['w'];
			}
			
			// output to buffer
			$this->out($resource, sprintf('q %.2f 0 0 %.2f %.2f %.2f cm /I%d Do Q', $width, $height, $x, $this->files[$resource]['h'] - ($y + $height), $info['i']));
		}
		
		/**
		 * Allows the setting of the font family, style and size to use until changed.
		 * @param int $resource is the resource of the document to use
		 * @param string $family is the font family to use
		 * @param string $style is the font style to use
		 * @param int $size is the font size to use
		 * @return void
		 * @access public
		 */
		public function setFont($resource, $family, $style = '', $size = null) {
			// make sure our resource is valid
			if($this->checkResource($resource) === FALSE) {
				throw new Exception($this->exception);
				return FALSE;
			}
			
			// make sure we have a good family
			$family = strtolower($family);
			if ($family == 'arial') {
				$family = 'helvetica';
			} else if ($family == 'symbol' || $family == 'zapfdingbats') {
				$style = '';
			}
			
			// check our style
			$style = strtoupper($style);
			if($style == 'IB') {
				$style = 'BI';
			}
			
			// check our size
			if(is_null($size)) {
				if(!$this->_font_size) {
					$size = 8;
				} else {
					$size = $this->_font_size;
				}
			}
			
			// set font key
			$fontkey = $family . $style;
			
			// check our font cache
			if(!isset($this->files[$resource]['fonts'][$fontkey])) {
				$i = count($this->files[$resource]['fonts']) + 1;
				$this->files[$resource]['fonts'][$fontkey] = Array(
					'i'		=>	$i,
					'name'	=>	$this->_core_fonts[$fontkey]
				);
			}
			
			// store current font information
			$this->_font_family = $family;
			$this->_font_style = $style;
			$this->_font_size = $size;
			$this->_current_font = $this->files[$resource]['fonts'][$fontkey];
			
			// load our font data set
			$this->loadFont($family, $style);
			
			// if we have a page, output font information
			if($this->files[$resource]['page'] > 0) {
				$this->out($resource, sprintf('BT /F%d %.2f Tf ET', $this->_current_font['i'], $this->_font_size));
			}
		}
		
		/**
		 * Allows you to set the font size to use until changed.
		 * @param int $resource is the resource id of the document to use
		 * @param int $size is the font size to use
		 * @return void
		 * @access public
		 */
		public function setFontSize($resource, $size) {
			// make sure our resource is valid
			if($this->checkResource($resource) === FALSE) {
				throw new Exception($this->exception);
				return FALSE;
			}
			
			// if the current font size is this font size, return
			if($this->_font_size == $size) {
				return;
			}
			
			// set our font size
			$this->_font_size = $size;
			
			// if we have a page, output font information
			if($this->files[$resource]['page'] > 0) {
				$this->out($resource, sprintf('BT /F%d %.2f Tf ET', $this->_current_font['i'], $this->_font_size));
			}
		}
		
		/**
		 * This method allows you to specify a bookmark at the current location in a document.
		 */
		public function addBookMark($resource) {
			// make sure our resource is valid
			if($this->checkResource($resource) === FALSE) {
				throw new Exception($this->exception);
				return FALSE;
			}
		}
		
		/**
		 * This method allows you to set a specific character spacing value to be used when writing text from this point forward.
		 * @param float $spacing is the spacing to use
		 * @return void
		 * @access public
		 */
		public function setCharacterSpacing($spacing) {
			// make sure the provided data is good 
			if(!is_numeric($spacing)) return FALSE;
			
			// set this value and return true 
			$this->_char_spacing = $spacing;
		}
		
		/**
		 * This method allows you to retrieve the current character spacing.
		 * @return float
		 * @access public
		 */
		public function getCharacterSpacing() {
			return $this->_char_spacing;
		}
		
		/**
		 * This method allows you to set a specific word spacing value to be used when writing text from this point forward.
		 * @param float $spacing is the spacing to use
		 * @return void
		 * @access public
		 */
		public function setWordSpacing($spacing){
			// make sure the provided data is good
			if(!is_numeric($spacing)) {
					$this->setException(1010);
					throw new Exception($this->exception);
					return FALSE;
			}
			
			// set this value and return true
			$this->_word_spacing = $spacing;
		}
		
		/**
		 * This method allows you to retrieve the current word spacing.
		 * @return float
		 * @access public
		 */
		public function getWordSpacing() {
			return $this->_word_spacing;
		}
		
		/**
		 * This method allows you to add a block of text that has each word offset above or below a horizontal
		 * rule. Angle and render values are the same as those used for {@link addText}.
		 * @param int $resource is the resource id of the document to use
		 * @param int $x is the x-coordinate of the starting position of the text
		 * @param int $y is the y-coordinate of the starting position of the text
		 * @param string $text is the text string to add
		 * @param int $deviation is the number of points to deviate each word from the horizontal
		 * @param int $wrap allows you to specify a character count to wordwrap this text on
		 * @param float $angle is the angle at which to place the text
		 * @param int $render allows you to modify the render mode used with this text. For possible values see {@link addText}.
		 * @return boolean
		 * @access public
		 */		
		public function addTextStitching($resource, $x, $y, $text, $wrap = 0, $angle = 0, $render = 0, $deviation = 0) {
			// make sure our resource is valid
			if($this->checkResource($resource) === FALSE) {
				throw new Exception($this->exception);
				return FALSE;
			}
			
			// if we have no local deviation but our global is set, we use that
			if(!$deviation && $this->_text_deviation) $deviation = $this->_text_deviation;
						
			// escape both text's
			$text = $this->escapeText($text);
			$subscript = $this->escapeText($subscript);
			
			// check our render mode
			$render = $this->checkTextRenderMode($render);
			
			// do we need to wrap this text? 
			if(!$wrap || $wrap === 0 || !is_numeric($wrap)) {
				// begin buffering font
				$this->out($resource, 'BT');
				
				// break this text string up by spaces
				$words = explode(' ', $text);
				
				// if this isn't angeled, its very easy
				if(!$angle || $angle === 0) {
					$this->out($resource, sprintf('%.2f %.2f Td', $x, ($this->files[$resource]['h'] - $y)));
				} else {
					$a = deg2rad((float)$angle);
					$this->out($resource, sprintf('%.2f %.2f %.2f %.2f %.2f %.2f Tm', cos($a), (-1.0 * sin($a)), sin($a), cos($a), $x, ($this->files[$resource]['h'] - $y)));
				}
				
				// modify our render mode and character spacing if we have to
				if($render) $out = $render . ' Tr ';
				if($this->_char_spacing) $out .= $this->_char_spacing . ' Tc ';
				if($out) $this->out($resource, $out);

				// loop through each word
				$i = 0;
				foreach($words as $word) {
					// trim to be safe
					$word = trim($word);
	
					// display this font
					$this->out($resource, sprintf('(%s ) Tj', $word));
					
					// do we need to add a deviation?
					if(($i % 2) == 0) {
						$this->out($resource, $deviation . ' Ts');
					} else {
						$this->out($resource, (0 - $deviation) . ' Ts');
					}
					$i++;
				}
				
				// end buffering
				$this->out($resource, 'ET');
				return TRUE;
			}
			
			$this->wrapText($resource, $x, $y, $text, $wrap, $angle, $render, 'addTextStitching', $deviation);
			return TRUE;
		}
		
		/**
		 * This method allows you to add a block of text that has a subscript attached to the end
		 * of it. Angle and render values are the same as those used for {@link addText}.
		 * @param int $resource is the resource id of the document to use
		 * @param int $x is the x-coordinate of the starting position of the text
		 * @param int $y is the y-coordinate of the starting position of the text
		 * @param string $text is the text string to add
		 * @param string $subscript is the subscript to add to the text
		 * @param float $angle is the angle at which to place the text
		 * @param int $render allows you to modify the render mode used with this text. For possible values see {@link addText}.
		 * @return boolean
		 * @access public
		 */
		public function addSubScript($resource, $x, $y, $text, $subscript, $angle = 0, $render = 0) {
			// make sure our resource is valid
			if($this->checkResource($resource) === FALSE) {
				throw new Exception($this->exception);
				return FALSE;
			}
			
			// escape both text's
			$text = $this->escapeText($text);
			$subscript = $this->escapeText($subscript);
			
			// check our render mode
			$render = $this->checkTextRenderMode($render);
			
			// begin buffering font
			$this->out($resource, 'BT');

			// if this isn't angeled, its very easy
			if(!$angle || $angle === 0) {
				$out = sprintf('%.2f %.2f Td', $x, ($this->files[$resource]['h'] - $y));
			} else {
				$a = deg2rad((float)$angle);
				$out = sprintf('%.2f %.2f %.2f %.2f %.2f %.2f Tm', cos($a), (-1.0 * sin($a)), sin($a), cos($a), $x, ($this->files[$resource]['h'] - $y));
			}
			$out .= ' ' . $render . ' Tr';
			if($this->_char_spacing) $out .= ' ' . $this->_char_spacing . ' Tc';
			if($this->_word_spacing) $out .= ' ' . $this->_word_spacing . ' Tw';
			$out .= sprintf(' (%s) Tj', $text);
			$out .= ' -5 Ts';
			$out .= sprintf(' (%s) Tj', $subscript);
			$this->out($resource, $out);
			
			// end bufferign font
			$this->out($resource, 'ET');
			return TRUE;
		}
		
		/**
		 * This method allows you to add a block of text that has a superscript attached to the end
		 * of it. Angle and render values are the same as those used for {@link addText}.
		 * @param int $resource is the resource id of the document to use
		 * @param int $x is the x-coordinate of the starting position of the text
		 * @param int $y is the y-coordinate of the starting position of the text
		 * @param string $text is the text string to add
		 * @param string $superscript is the superscript to add to the text
		 * @param float $angle is the angle at which to place the text
		 * @param int $render allows you to modify the render mode used with this text. For possible values see {@link addText}.
		 * @return boolean
		 * @access public
		 */
		public function addSuperScript($resource, $x, $y, $text, $superscript, $angle = 0, $render = 0) {
			// make sure our resource is valid
			if($this->checkResource($resource) === FALSE) {
				throw new Exception($this->exception);
				return FALSE;
			}
			
			// escape both text's
			$text = $this->escapeText($text);
			$superscript = $this->escapeText($superscript);
			
			// check our render mode
			$render = $this->checkTextRenderMode($render);
			
			// begin buffering font
			$this->out($resource, 'BT');

			// if this isn't angeled, its very easy
			if(!$angle || $angle === 0) {
				$out = sprintf('%.2f %.2f Td', $x, ($this->files[$resource]['h'] - $y));
			} else {
				$a = deg2rad((float)$angle);
				$out = sprintf('%.2f %.2f %.2f %.2f %.2f %.2f Tm', cos($a), (-1.0 * sin($a)), sin($a), cos($a), $x, ($this->files[$resource]['h'] - $y));
			}
			$out .= ' ' . $render . ' Tr';
			if($this->_char_spacing) $out .= ' ' . $this->_char_spacing . ' Tc';
			if($this->_word_spacing) $out .= ' ' . $this->_word_spacing . ' Tw';
			$out .= sprintf(' (%s) Tj', $text);
			$out .= ' 5 Ts';
			$out .= sprintf(' (%s) Tj', $superscript);
			$this->out($resource, $out);
			
			// end bufferign font
			$this->out($resource, 'ET');
			return TRUE;
		}
		
		/**
		 * This method allows you to add test to a document. When placing text at an angle, 
		 * the angle is measured clock-wise around the (x, y) coordinate along the x-axis.
		 * @param int $resource is the resource id of the document to use
		 * @param int $x is the x-coordinate of the starting position of the text
		 * @param int $y is the y-coordinate of the starting position of the text
		 * @param string $text is the text string to add
		 * @param int $wrap is how many characters to word wrap the text on
		 * @param float $angle is the angle at which to place the text
		 * @param int $render allows you to modify the render mode used with this text. Possible values are<br>
		 * '0' - Fill text (<i>default</i>)<br>
		 * '1' - Stroke text <br>
		 * '2' - Fill and stroke text <br>
		 * '3' - Neither fill nor stroke text (invisible) <br>
		 * '4' - Fill text and add to path for clipping <br>
		 * '5' - Stroke text and add to path for clipping <br>
		 * '6' - Fill, then stroke text, and add to path for clipping <br>
		 * '7' - Add text to path for clipping
		 * @return boolean
		 * @access public
		 */
		public function addText($resource, $x, $y, $text, $wrap = 0, $angle = 0, $render = 0) {
			// make sure our resource is valid
			if($this->checkResource($resource) === FALSE) {
				throw new Exception($this->exception);
				return FALSE;
			}
			
			// escape our text
			$text = $this->escapeText($text);
			
			// if we have a render mode, make sure it is good
			if($render !== 0) $render = $this->checkTextRenderMode($render);
			
			// make sure we can wrap
			if(!strpos($text, ' ') || !strpos($text, '-')) $wrap = 0;
			
			// if we don't have a wrap, just display our text
			if(!$wrap || $wrap === 0 || !is_numeric($wrap)) {
				// begin buffering font
				$this->out($resource, 'BT');
				
				// if this isn't angeled, its very easy
				if(!$angle || $angle === 0) {
					$out = sprintf('%.2f %.2f Td', $x, ($this->files[$resource]['h'] - $y));
				} else {
					$a = deg2rad((float)$angle);
					$out = sprintf('%.2f %.2f %.2f %.2f %.2f %.2f Tm', cos($a), (-1.0 * sin($a)), sin($a), cos($a), $x, ($this->files[$resource]['h'] - $y));
				}
				$out .= ' ' . $render . ' Tr';
				if($this->_char_spacing) $out .= ' ' . $this->_char_spacing . ' Tc';
				if($this->_word_spacing) $out .= ' ' . $this->_word_spacing . ' Tw';
				$out .= sprintf(' (%s) Tj', $text);
				$this->out($resource, $out);

				// end font buffering
				$this->out($resource, 'ET');
				return TRUE;
			}
			
			$this->wrapText($resource, $x, $y, $text, $wrap, $angle, $render, 'addText');
			return TRUE;
		}
		
		/**
		 * This method allows the adding of both links and internal links. A link allows 
		 * to place an anchor that goes to a separate location while an internal link allows
		 * for an anchor that goes to another location within the document.
		 * @param int $resource is the resource id of the document to use
		 * @param int $x1 is the x-coordinate of the top left corner of the link
		 * @param int $y1 is the y-coordinate of the top left corner of the link
		 * @param int $x2 is the x-coordinate of the bottom right corner of the link
		 * @param int $y2 is the y-coordinate of the bottom right corner of the link
		 * @return void
		 * @access public
		 */
		public function addLink($resource, $x1, $y1, $x2, $y2, $url) {
			// make sure our resource is valid
			if($this->checkResource($resource) === FALSE) {
				throw new Exception($this->exception);
				return FALSE;
			}
			
			// adjust our y's
			$y1 = $this->files[$resource]['h'] - $y1;
			$y2 = $this->files[$resource]['h'] - $y2;
			
			// add this link to our buffer
			$this->newObj($resource);
			$this->out($resource, '<<');
			$this->out($resource, '/Type /Annot');
			$this->out($resource, '/Subtype /Text');
			$this->out($resource, "/Rect [$x1, $x2, $y1, $y2]");
			$this->out($resource, "/Contents ($url)");
			$this->out($resource, '>>');
			$this->out($resource, 'endobj');
		}
		
		/**
		 * This method initiates the drawing of a table by creating it with a data
		 * set and setting the beginning coords and setting any default settings.
		 * @param int $resource is the resource id of the document to use
		 * @param string $name is the name of this table, used for later reference
		 * @param int $x is the x-coordinate of the upper left hand corner of the table
		 * @param int $y is the y-coordinate of the upper left hand corner of the table
		 * @param array $data is the beginning data set of the table. The data is an array of 
		 * columns, each column being an array of data within that column. <b>Example</b><br />
		 * Array('col1' => Array(1,2,3,4,5), 'col2' => Array(6,7,8,9,0))
		 * @param array $settings is a list of default settings used by the table. Available settings 
		 * include<br /><i>cellspacing</i> The amount of space between cells<br /><i>cellpadding</i> The
		 * amount of space between the data of a cell and it's borders
		 * @return void
		 * @access public
		 */
		public function createTable($resource, $name, $x, $y, $data, $settings = Array()) {
			// make sure our resource is valid
			if($this->checkResource($resource) === FALSE) {
				throw new Exception($this->exception);
				return FALSE;
			}
			
			// build starting structure for table
			$newTable = Array();
			$newTable['data'] = $data;
			$newTable['x'] = $x;
			$newTable['y'] = $y;
			$newTable['page'] = $this->files[$resource]['page'];
			$newTable['settings'] = $settings;
			
			// initiate this table array in our file list
			$this->files[$resource]['tables'][$name] = $newTable;
			
			// make sure we have a font size since it's being used below
			if(!$this->_font_size) $font_size = 2;
			else $font_size = $this->_font_size + 1;
			
			// now we need to build a list of all column names we need
			$columns = Array();
			foreach($data as $key => $value) {
				// begin column array
				$col = Array('name' => trim($key), 'width' => strlen(trim($key)), 'data' => $value);
				
				// first set width to the length of the header
				$col['width'] = $this->getStringLength(trim($key));
				
				// determine longest field in column
				foreach($value as $k => $v) {
					$v = trim($v);
					if($this->getStringLength($v) > $col['width']) $col['width'] = $this->getStringLength($v);
				}
				
				// assign column to list
				$this->files[$resource]['tables'][$name]['cols'][trim($key)] = $col;
			}
		}
		
		/**
		 * This method gets the length of a string as it will be used in the document.
		 * @param string $string is the string to measure.
		 * @return int
		 * @access public
		 */
		public function getStringLength($string) {
			// make sure we have font data, if not - try to load it
			if(!$this->_font_data || !count($this->_font_data)) {
				if(!$this->loadFont($this->_font_family, $this->_font_style)) return strlen($string);
			}
			
			// we have a font data set, so what we do now is go through each character
			// of our string and find it in the data set and record the length
			$length = 0;
			$string_len = strlen($string);
			if(($type = gettype($string)) != 'string') {
				settype($string, 'string');
			}
			for($i = 0; $i < $string_len; $i++) {
				if(!$this->_font_data[$string{$i}]) continue;
				$length += $this->_font_data[$string{$i}] / 100;
			}
			return $length;
		}
		
		/**
		 * This method sets an internal var for a table, these vars allow you to set how certain things
		 * in the table are handled.
		 * @param int $resource is the resource id of the document to use
		 * @param string $name is the name of the table to use
		 * @param string $var is the name of the var to set
		 * @param string $value is the value to give to the var
		 * @return void
		 * @access public
		 */
		public function setTableVars($resource, $name, $var, $value) {
			// make sure our resource is valid
			if($this->checkResource($resource) === FALSE) {
				throw new Exception($this->exception);
				return FALSE;
			}

			// make sure our table exists
			if(!$this->files[$resource]['tables'][$name]) {
				$this->setException(1011);
				throw new Exception($this->exception);
				return FALSE;
			}
			
			// set this var
			$this->files[$resource]['tables'][$name]['vars'][$var] = $value;
		}
		
		/**
		 * This method sets an internal setting for a table. These settings control how the table will be drawn.
		 * @param int $resouse is the resource id of the document to use
		 * @param string $name is the name of the table to use
		 * @param string $setting is the name of the setting to change
		 * @param string $value is the value to use
		 * @return void
		 * @access public
		 */
		public function setTableSettings($resource, $name, $setting, $value) {
			// make sure our resource is valid
			if($this->checkResource($resource) === FALSE) {
				throw new Exception($this->exception);
				return FALSE;
			}

			// make sure our table exists
			if(!$this->files[$resource]['tables'][$name]) {
				$this->setException(1011);
				throw new Exception($this->exception);
				return FALSE;
			}
			
			// set this setting
			$this->files[$resource]['tables'][$name]['settings'][$setting] = $value;
		}
				
		/**
		 * This method writes a table to the document.
		 * @param int $resource is the resource id of the document to use
		 * @param string $name is the name of the table to write.
		 * @return void
		 * @access public
		 */
		public function writeTable($resource, $name) {
			// make sure our resource is valid
			if($this->checkResource($resource) === FALSE) {
				throw new Exception($this->exception);
				return FALSE;
			}
			
			// make sure our table exists
			if(!$this->files[$resource]['tables'][$name]) {
				$this->setException(1011);
				throw new Exception($this->exception);
				return FALSE;
			}
			
			// make sure we have columns
			if(!is_array($this->files[$resource]['tables'][$name]['cols'])) {
				$this->setException(1012);
				throw new Exception($this->exception);
				return FALSE;
			}
			
			// set our page pointer
			$this->setPagePointer($resource, $this->files[$resource]['tables']['page']);
			
			// store our settings in more easily accessible variables
			foreach((array)$this->files[$resource]['tables'][$name]['settings'] as $sname => $value) {
				$sname = '_'.$sname;
				${$sname} = $value;
			}

			// check some of our settings for data integrity
			$num_checks = Array('_cellspacing', '_cellpadding', '_border');
			foreach($num_checks as $check) {
				if(!${$check} && ${$check} !== 0) ${$check} = 0;
				if($$check > 0 && !is_numeric(${$check})) ${$check} = 0;
			}

			// if we have a border we need to backup our current line width and set a new one
			$line_width = FALSE;
			$draw = FALSE;
			$fill = FALSE;
			if($_border) {
				// backup our line width
				$line_width = $this->getLineWidth($resource);

				// set our new line width
				$this->setLineWidth($resource, $_border);
				
				// if we have a border color, convert it to rgb and set it
				if($_bordercolor) {
					$_draw = $this->_draw_color;
					list($r, $g, $b) = $this->hex2rgb($_bordercolor);
					$this->setDrawColor($resource, 'rgb', $r, $g, $b);
				}
			}
			
			// now we need to draw our column headers
			$useX = $this->files[$resource]['tables'][$name]['x'];
			$useY = $this->files[$resource]['tables'][$name]['y'];
			
			// default our iterators
			$counter_cols = 0;
			$counter_rows = 0;
			
			// loop through our columns
			foreach($this->files[$resource]['tables'][$name]['cols'] as $cname => $info) {
				// write our column header and offset our x-coordinate accordingly
				// if we have a header var to modify the header text we need to apply it now
				// default some vars
				$this_style = false;
				$old_style = false;
				switch($this->files[$resource]['tables'][$name]['vars']['headers']) {
					case 'bold':
					case 'b':
						if(!$this_style) $this_style = 'b';
					case 'italic':
					case 'i':
						if(!$this_style) $this_style = 'i';
					case 'ib':
					case 'bi':
						if(!$this_style) $this_style = 'bi';
						// store our old style
						$old_style = $this->_font_style;
						
						// change our current font style
						$this->setFont($resource, $this->_font_family, $this_style, $this->_font_size);
				}
				
				// write this header name
				$this->addText($resource, ($useX + $_cellpadding), ($useY + $_cellpadding), $cname, $info['width']);
				
				// did we change our style?
				if($this_style) {
					// we did, set it back
					$this->setFont($resource, $this->_font_family, $old_style, $this->_font_size);
				}
				
				// if we have a border then we need to draw lines around our column headers
				// bottom, top, left, right 
				if($_border) {
					// bottom
					$this->drawLine($resource, ($useX - $_cellpadding + ($_cellspacing * $counter_cols)), ($useY + ($_border * 1.25) + $_cellpadding), ($useX + ($info['width'] + $this->_font_size + ($_cellspacing * $counter_cols)) + $_cellpadding), ($useY + ($_border * 1.25) + $_cellpadding));
					
					// top
					$this->drawLine($resource, ($useX - $_cellpadding + ($_cellspacing * $counter_cols)), ($useY - $this->_font_size - $_border - $_cellpadding), ($useX + ($info['width'] + $this->_font_size) + $_cellpadding + ($_cellspacing * $counter_cols)), ($useY - $this->_font_size - $_border - $_cellpadding));

					// left
					$this->drawLine($resource, ($useX - $_cellpadding + ($_cellspacing * $counter_cols)), ($useY + ($_border * 1.25) + $_cellpadding), ($useX - $_cellpadding + ($_cellspacing * $counter_cols)), ($useY - $this->_font_size - ($_border * 1.25) - $_cellpadding));

					// right
					$this->drawLine($resource, ($useX + ($info['width'] + $this->_font_size) + $_cellpadding + ($_cellspacing * $counter_cols)), ($useY + ($_border * 1.25) + $_cellpadding), ($useX + ($info['width'] + $this->_font_size) + $_cellpadding + ($_cellspacing * $counter_cols)), ($useY - $this->_font_size - ($_border * 1.25) - $_cellpadding));
					
					// increment our counter
					$counter_cols++;
				}

				// offset our x for the next header
				$useX += ($info['width'] + $this->_font_size);

				// if we have a cellspacing, use it here
				if($_cellspacing > 0 && is_numeric($_cellspacing)) {
					$useX += $_cellspacing;
				}
			}
			
			// now we need to draw our cells
			$useX = $this->files[$resource]['tables'][$name]['x'];
			foreach($this->files[$resource]['tables'][$name]['cols'] as $cname => $info) {
				$useY = $this->files[$resource]['tables'][$name]['y'] + $this->_font_size + $_cellspacing;
				foreach($info['data'] as $cell) {
					// write our data to our cell
					$this->addText($resource, $useX, $useY, $cell, $info['width']);

					// if we have a border then we need to draw lines around our column headers
					if($_border) {
						$this->drawLine($resource, ($useX - $_cellpadding), ($useY + $_border + $_cellpadding), ($useX + $info['width'] + $this->_font_size + $_cellpadding), ($useY + $_border + $_cellpadding));
						$this->drawLine($resource, ($useX - $_cellpadding), ($useY + $_border + $_cellpadding), ($useX - $_cellpadding), ($useY - $this->_font_size - $_border - $_cellpadding));
						$this->drawLine($resource, ($useX + ($info['width'] + $this->_font_size) + $_cellpadding), ($useY + $_border + $_cellpadding), ($useX + $info['width'] + $this->_font_size + $_cellpadding), ($useY - $this->_font_size - $_border - $_cellpadding));
					}

					// offset our y for the next cell
					$useY += $this->_font_size + $_cellspacing;
				}
	
				// offset our x for the next column
				$useX += ($info['width'] + $this->_font_size);
				
				// if we have a cellspacing, use it here
				if($_cellspacing > 0 && is_numeric($_cellspacing)) {
					$useX += $_cellspacing;
				}
			}
			
			// if we have a border, reset our line width
			if($_border) {
				$this->setLineWidth($resource, $line_width);
			}
			
			// if we have a fill color, reset it
			if($fill) {
				$this->setFillColor($resource, $fill['cs'], $fill['cs1'], $fill['cs2'], $fill['cs3'], $fill['cs4']);
			}
			
			// if we have a draw color, reset it
			if($draw) {
				$this->setDrawColor($resource, $draw['cs'], $draw['cs1'], $draw['cs2'], $draw['cs3'], $draw['cs4']);
			}
			
			// reset our page pointer
			$this->resetPagePointer($resource);
		}
		
		/**
		 * Allows the adding of a value to the end of column in a table
		 * @param int $resource is the resource id of the document to use
		 * @param string $name is the name of the table to use 
		 * @param mixed $column is the column to add the data to
		 * @param mixed $data is the value to add onto the column
		 * @return void
		 * @access public
		 */
		public function addTableData($resource, $name, $column, $data) {
			// make sure our resource is valid
			if($this->checkResource($resource) === FALSE) {
				throw new Exception($this->exception);
				return FALSE;
			}
			
			// make sure our table exists
			if(!$this->files[$resource]['tables'][$name]) {
				$this->setException(1011);
				throw new Exception($this->exception);
				return FALSE;
			}
			
			// make sure this column exists
			if(!$this->files[$resource]['tables'][$name]['cols'][$column]) {
				$this->files[$resource]['tables'][$name]['cols'][$column] = Array('name' => $column);
			}
			
			// loop through this data adding each entry to this table
			if(!is_array($data)) $this->files[$resource]['tables'][$name]['cols'][$column]['data'][] = $data;
			else {
				foreach($data as $row) {
					$this->files[$resource]['tables'][$name]['cols'][$column]['data'][] = $row;
				}
			}
			
			// now we need to recalculate the width of this column
			$this->setColumnWidth($resource, $name, $column, $this->getColumnWidth($resource, $name, $column));
		}

		/**
		 * Forces a width to be used when writing a column. If a width is not manually set
		 * then one is automatically calculated based on the data in the column.
		 * @param int $resource is the resource id of the document to use
		 * @param string $name is the name of the table to use
		 * @param string $column is the column to use
		 * @param int $width is the width to use for the column
		 * @return void
		 * @access public
		 */
		public function setColumnWidth($resource, $name, $column, $width) {
			// make sure our resource is valid
			if($this->checkResource($resource) === FALSE) {
				throw new Exception($this->exception);
				return FALSE;
			}
			
			// is our width even numeric?
			if(!is_numeric($width)) return FALSE;
			
			// does this table exist?
			if(!$this->files[$resource]['tables'][$name]) {
				$this->setException(1011);
				throw new Exception($this->exception);
				return FALSE;
			}
			
			// does this column exist?
			if(!$this->files[$resource]['tables'][$name]['cols'][$column]) {
				return;
			}

			// set this width
			$this->files[$resource]['tables'][$name]['cols'][$column]['width'] = $width;
			return TRUE;
		}

		/**
		 * Sets a table columns width.
		 * @param int $resource is the resource if od the document to use
		 * @param string $name is the name of the table to use
		 * @param string $column is the column to use
		 * @return int
		 * @access private
		 */		
		private function getColumnWidth($resource, $name, $column) {
			// make sure our resource is valid
			if($this->checkResource($resource) === FALSE) {
				throw new Exception($this->exception);
				return FALSE;
			}
			
			// make sure our table exists
			if(!$this->files[$resource]['tables'][$name]) {
				$this->setException(1011);
				throw new Exception($this->exception);
				return FALSE;
			}
			
			// make sure this column exists in this table, if it doesn't
			// just return 0, we wont throw an error for these
			if(!$this->files[$resource]['tables'][$name]['cols'][$columns]) {
				return 0;
			}
			
			// first set width to the length of the header
			$width = $this->getStringLength(trim($name));
			
			// determine longest field in column
			foreach($value as $k => $v) {
				$v = trim($v);
				if($this->getStringLength($v) > $width) $width = $this->getStringLength($v);
			}
			
			// assign column to list
			return $width;	
		}

		/**
		 * Loads a fonts data file
		 * @param string $family is the name of the font family to load
		 * @param string $style is the style of the font to load
		 * @return void
		 * @access private
		 */
		private function loadFont($family, $style = null) {
			// concatenate our family and style
			if(!is_null($style)) $name = $family . $style;
			else $name = $family;
			$name = strtolower($name);
			$_name = strtolower($family) . strtoupper($style);
			
			// see if our file exists
			if(!@file_exists('./fonts/'.$name.'.php')) {
				// do we have a style? if so try dropping it
				if(!is_null($style)) {
					$name = strtolower($family);
					$_name = $name;
					if(!@file_exists('./fonts/'.$name.'.php')) {
						$this->setException(1013);
						throw new Exception($this->exception);
						return FALSE;
					}
				} else {
					$this->setException(1013);
					throw new Exception($this->exception);
					return FALSE;
				}
			}
			
			// file exists, load it
			@include_once('./fonts/'.$name.'.php');
			$this->_font_data = $pdfclass_fonts[$_name];
		}

		/**
		 * Rewrites a string wrapped to the provided character length
		 * @param int $resource is the resource of the document to use
		 * @param int $x is the x-coordinate to start the text at
		 * @param int $y is the y-coordinate to start the text at
		 * @param string $text is the text to wrap
		 * @param int $wrap is the character length to wrap the text at
		 * @param int $angle is the angle the text is at
		 * @param string $render is the render mode of the text
		 * @param string $method is the method to call to write the text
		 * @param int $deviant is the vertical deviation of the text
		 * @return bool
		 * @access private
		 */
		private function wrapText($resource, $x, $y, $text, $wrap, $angle, $render, $method, $deviation = 0) {
			// if we have a local deviation but no global, we set the global
			if($deviation && !$this->_text_deviation) $this->_text_deviation = $deviation;
			
			// if we have no local deviation but we have our global, use our global
			if(!$deviation && $this->_text_deviation) $deviation = $this->_text_deviation;
			
			// set some default values
			$length = strlen($text);
			$tmpString = '';
			$break = 0;
			$startY = $y;
			
			// begin looping through each letter
			for($i = 0; $i < $length; $i++) {
				// build our temporary string
				$tmpString .= (string)$text{$i};
				
				// can we break at this location if we need to?
				if((string)$text{$i} == '-' || (string)$text{$i} == ' ') {
					$break = strlen($tmpString);
				}
				
				// should we break at our last saved location?
				if(strlen($tmpString) == $wrap) {
					// our string is at our limit, we need to break
					$string = substr($tmpString, 0, $break);
					$tmpString = substr($tmpString, $break, strlen($tmpString) - 1);
					$this->{$method}($resource, $x, $startY, $string, 0, $angle, $render);
					$startY += $this->_font_size;
					if($deviation > 0) $startY += ($deviation * 2);
				}
			}
			
			// make sure our global deviation is unset before ending this function
			$this->_text_deviation = 0;
			return TRUE;		
		}
	
		/**
		 * Ensures the provided render mode is valid, if not then a defaulted value will be returned
		 * @param int $mode is the render mode to check
		 * @return int
		 * @access private
		 */
		private function checkTextRenderMode($mode) {
			if(!is_numeric($mode)) $mode = 0;
			else {
				if($mode < 0) $mode = 0;
				if($mode > 7) $mode = 0;
			}
			return $mode;
		}
	
		/**
		 * @access private
		 */
		private function parseJPEG($file) {
			// extract info
			$img = @getimagesize($file);
			if(!$img) {
				$this->setException(4002);
				throw new Exception($this->exception);
				return FALSE;
			}
			
			// check if dealign with an actual jpeg
			if($img[2] != 2) {
				$this->setException(4001);
				throw new Exception($this->exception);
				return FALSE;
			}
			
			// get the image colorspace
			if(!isset($img['channels']) || $img['channels'] == 3) {
				$colspace = 'DeviceRGB';
			} else if($img['channels'] == 4) {
				$colspace = 'DeviceCMYK';
			} else {
				$colspace = 'DeviceGray';
			}
			$bpc = isset($img['bits']) ? $img['bits'] : 8;
			
			// read the whole file
			$f = fopen($file, 'rb');
			$data = fread($f, filesize($f));
			fclose($f);
			
			// return our data
			return Array('w' => $img[0], 'h' => $img[1], 'cs' => $colspace, 'bpc' => $bpc, 'f' => 'DCTDecode', 'data' => $data);
		}
		
		/**
		 * @access private
		 */
		private function checkResource($resource) {
			if(!is_numeric($resource)) {
				$this->setException(1004);
				return;
			}

			if(!is_array($files[$resource])) {
				$this->setException(1004);
				return;
			}	
		}
		
		/**
		 * @access private
		 */
		private function checkFileName($name) {
			foreach($this->files as $index => $file) {
				if($file['name'] == $name) return TRUE;
			}
			return FALSE;
		}
		
		/**
		 * @access private
		 */
		private function getResourceId($name) {
			foreach($this->files as $index => $file) {
				if($file['name'] == $name) return $index;
			}
			return FALSE;
		}
		
		/**
		 * @access private
		 */
		private function setException($errno) {
			$this->exception = $this->_error_codes[$errno];
			$this->errno = $errno;
		}
		
		/**
		 * @access private
		 */
		private function out($resource, $buffer) {
			// make sure our resource is valid
			If($this->checkResource($resource) === FALSE) {
				throw new Exception($this->exception);
				return FALSE;
			}
			
			// end buffering based on document state
			if($this->files[$resource]['state'] == 2) {
				$this->files[$resource]['pages'][$this->files[$resource]['page']] .= $buffer . "\n";
			} else {
				$this->files[$resource]['buffer'] .= $buffer . "\n";
			}
		}
		
		/**
		 * @access private
		 */
		private function escapeText($text) {
			$text = str_replace('\\', '\\\\', $text);
			$text = str_replace('(', '\\(', $text);
			$text = str_replace(')', '\\)', $text);
			return $text;
		}
		
		/**
		 * @access private
		 */
		private function newobj($resource) {
			// make sure our resource is valid
			If($this->checkResource($resource) === FALSE) {
				throw new Exception($this->exception);
				return FALSE;
			}
			
			// increment object counter
			$this->files[$resource]['n']++;
			
			// save the byte offset of this object
			$this->files[$resource]['offsets'][$this->files[$resource]['n']] = strlen($this->files[$resource]['buffer']);
			
			// output to buffer
			$this->out($resource, $this->files[$resource]['n'] . ' 0 obj');
		}
		
		/**
		 * @access private
		 */
		private function putVars($resource) {
			// make sure we have vars
			if(!$this->files[$resource]['vars'] || !count($this->files[$resource]['vars'])) return;
			
			// we have at least one, make sure we have an array
			if($this->files[$resource]['vars'] && !is_array($this->files[$resource]['vars'])) {
				$this->files[$resource]['vars'] = Array($this->files[$resource]['vars']);
			}
			
			// loop through each var
			$obStart = FALSE;
			foreach($this->files[$resource]['vars'] as $var => $value) {
				// we only care about certain vars here, others we disregard
				switch($var) {
			    case 'Title':
			    case 'Author':
			    case 'Subject':
			    case 'Keywords':
			    case 'Creator':
			    case 'Producer':
			    case 'CreationDate':
			    case 'ModDate':
			    case 'Trapped':
			    	if(!$obStart) {
			    		$this->newObj($resource);
			    		$this->out($resource, '<<');
			    		$obStart = TRUE;
			    	}
			    	$this->out($resource, '/' . $var . ' (' . $this->escapeText($value) . ')');
			    	break;
				}
			}
			if($obStart) {
  	  	$this->out($resource, '>>');
	    	$this->out($resource, 'endobj');			
			}
		}
		
		/**
		 * @access private
		 */
		private function putPages($resource) {
			// if compression is required, set the compression tag
			$filter = ($this->files[$resource]['compress']) ? '/Filter /FlatDecode ' : '';
			
			// print out pages
			for ($n = 1; $n <= $this->files[$resource]['page']; $n++) {
				$this->newobj($resource);
				$this->out($resource, '<</Type /Page');
				$this->out($resource, '/Parent 1 0 R');
				$this->out($resource, '/Resources 2 0 R');
				$this->out($resource, '/Contents ' . ($this->files[$resource]['n'] + 1) . ' 0 R>>');
				$this->out($resource, 'endobj');
				
				// if compression is required, gzcompress the page contents
				$p = ($this->files[$resource]['compress'] ? gzcompress($this->files[$resource]['pages'][$n]) : $this->files[$resource]['pages'][$n]);
				
				// output the page contents
				$this->newobj($resource);
				$this->out($resource, '<<' . $filter . '/Length ' . strlen($p) . '>>');
				$this->putStream($resource, $p);
				$this->out($resource, 'endobj');
			}
			
			// set the offset of the first object
			$this->files[$resource]['offsets'][1] = strlen($this->files[$resource]['buffer']);
			$this->out($resource, '1 0 obj');
			$this->out($resource, '<</Type /Pages');
			$kids = '/Kids [';
			for($i = 0; $i < $this->files[$resource]['page']; $i++) {
				$kids .= (3 + 2 * $i) . ' 0 R ';
			}
			$this->out($resource, $kids . ']');
			$this->out($resource, '/Count ' . $this->files[$resource]['page']);
			
			// output the page size
			$this->out($resource, sprintf('/MediaBox [0 0 %.2f %.2f]', $this->files[$resource]['w'], $this->files[$resource]['h']));
			$this->out($resource, '>>');
			$this->out($resource, 'endobj');
		}
		
		/**
		 * @access private
		 */
		private function putStream($resource, $stream) {
			// make sure our resource is valid
			If($this->checkResource($resource) === FALSE) {
				throw new Exception($this->exception);
				return FALSE;
			}
			
			// add stream to buffer
			$this->out($resource, 'stream');
			$this->out($resource, $stream);
			$this->out($resource, 'endstream');
		}
		
		/**
		 * @access private
		 */
		private function putResources($resource) {
			// make sure our resource is valid
			If($this->checkResource($resource) === FALSE) {
				throw new Exception($this->exception);
				return FALSE;
			}
			
			// output any fonts and images
			$this->putFonts($resource);
			$this->putImages($resource);
			
			// resoures are always object number 2
			$this->files[$resource]['offsets'][2] = strlen($this->files[$resource]['buffer']);
			$this->out($resource, '2 0 obj');
			$this->out($resource, '<</ProcSet [/PDF /Text /ImageB /ImageC /ImageI]');
			$this->out($resource, '/Font <<');
			foreach($this->files[$resource]['fonts'] as $font) {
				$this->out($resource, '/F' . $font['i'] . ' ' . $font['n'] . ' 0 R');
			}
			$this->out($resource, '>>');
			
			if(count($this->files[$resource]['images'])) {
				$this->out($resource, '/XObject <<');
				foreach($this->files[$resource]['images'] as $image) {
					$this->out($resource, '/I' . $image['i'] . ' ' . $image['n'] . ' 0 R');
				}
				$this->out($resource, '>>');
			}
			
			$this->out($resource, '>>');
			$this->out($resource, 'endobj');
		}
		
		/**
		 * @access private
		 */
		private function putImages($resource) {
			// output any images
			$file = ($this->files[$resource]['compress']) ? '/Filter /FlateDe ' : '';
			foreach($this->files[$resource]['images'] as $file => $info) {
				$this->newobj($resource);
				$this->files[$resource]['images'][$file]['n'] = $this->files[$resource]['images']['n'];
				$this->out($resource, '<</Type /XObject');
				$this->out($resource, '/Subtype	/Image');
				$this->out($resource, '/Width ' . $info['w']);
				$this->out($resource, '/Height ' . $info['h']);
				$this->out($resource, '/ColorSpace /' . $info['cs']);
				if($info['cs'] = 'DeviceCYMK') {
					$this->out($resource, '/De [1 0 1 0 1 0 1 0]');
				}
				$this->out($resource, '/BitsPerComponent ' . $info['bpc']);
				$this->out($resource, '/Filter /' . $info['f']);
				$this->out($resource, '/Length ' . strlen($info['data']) . '>>');
				$this->putStream($resource, $info['data']);
				$this->out($resource, 'endobj');
			}
		}
		
		/**
		 * @access private
		 */
		private function putFonts($resource) {
			// make sure our resource is valid
			If($this->checkResource($resource) === FALSE) {
				throw new Exception($this->exception);
				return FALSE;
			}
			
			// print out font details
			foreach($this->files[$resource]['fonts'] as $k => $font) {
				$this->newobj($resource);
				$this->files[$resource]['fonts'][$k]['n'] = $this->files[$resource]['n'];
				$name = $font['name'];
				$this->out($resource, '<</Type /Font');
				$this->out($resource, '/BaseFont /' . $name);
				$this->out($resource, '/Subtype /Typel');
				if ($name != 'Symbol' && $name != 'ZapfDingbats') {
					$this->out($resource, '/Encoding /WinAnsiEncoding');
				}
				$this->out($resource, '>>');
				$this->out($resource, 'endobj');
			}
		}
		
		/**
		 * @access private
		 */
		private function hex2rgb($color) {
			if($color{0} == '#') {
				$color = substr($color, 1);
			}
			
			if(strlen($color) == 6) {
				list($r, $g, $b) = Array($color{0}.$color{1}, $color{2}.$color{3}, $color{4}.$color{5});
			} else if(strlen($color) == 3) {
				list($r, $g, $b) = Array($color{0}, $color{1}, $color{2});
			} else {
				return FALSE;
			}
			
			$r = hexdec($r);
			$g = hexdec($g);
			$b = hexdec($b);
			
			return Array($r, $g, $b);
		}
	}
?>