<?php
/**
 * Copyright 2008 Ivan Ribas
 * Licensed under The MIT License
 * Redistributions of files must retain the above copyright notice.
 *
 * @author    Ivan Ribas
 * @copyright Copyright 2008 Ivan Ribas
 * @license    http://www.opensource.org/licenses/mit-license.php The MIT License
 */

uses('sanitize');

class EventsController extends AppController {

	var $name = 'Events';
	var $uses = array('Event', 'Tag', 'EventsTag', 'Country', 'City', 'User');
	var $helpers = array('Html', 'Form', 'Calendar', 'Time', 'Text', 'Ajax', 'Javascript', 'googleMap');
	var $components = array('RequestHandler', 'Email');
  var $paginate = array(
                        'limit' => 15,
                        'order' => array(
                        'Event.created' => 'desc'
                        )
                  );
	
   function beforeFilter() {
    $this->checkSession();
    $this->Auth->authorize = 'controller';
    $this->Auth->allow('index','view','autoCompleteTags','autoComplete', 'report');
    
    if($this->Auth->user('id')) {
      $this->Auth->allow('addAttendee','delAttendee', 'edit');
    }
    
    if(Configure::read('eventoSettings.allowAnonymousEvents') or $this->Auth->user('id')) 
    {
      $this->Auth->allow('add');
    }
   }

   /**
    * Admin can access all actions in this controller
    */
    
   function isAuthorized() {
      return $this->Auth->user('admin');
   }
   
	/**
	*
	* @param city string
	* @param tag string
	*	@param $year string
	* @param $month string
	* @param $day string
	*
	**/

	function index($city='all-cities',$tag='all-tags',$year = null, $month = null,$day = null) {
		$tag_id = $city_id = false;
		$city_name = 'all-cities';
		$data   = null;
		$feed = 'events';
		$month_list = array('january', 'febuary', 'march', 'april', 'may', 
		                    'june', 'july', 'august', 'september', 'october', 
		                    'november', 'december');
		                    
		$day_list = array('Mon', 'Tue', 'Wed', 'Thu', 'Fri', 'Sat', 'Sun');
		$base_url = 'index/'.$city.'/'.$tag;
		$view_base_url = $this->webroot. 'events';

		  
		if($city!='all-cities'):
		  $current_city = $this->City->find(array('City.slug'=>$city),array('City.id','City.name'),null,(-1));
		  if($current_city) {
		    $city_name = $current_city['City']['name'];
		    $city_id = $current_city['City']['id'];
		    $feed = $city;
		  }
		  else $city = 'all-cities';
		endif;
		  
		$this->set('feed',$feed);
		  
		if($tag!='all-tags'):
	    $tag_id = $this->Tag->field('id',array('Tag.slug'=>$tag));
	    if(!$tag_id) $tag = 'all-tags';
		endif;

		if(!($year>0)) $year = date('Y');	
		if(($month_num=array_search($month,$month_list))) {
		  $month_num++;
		}
		else {
				$month = date('M');
				$month_name = date('F');
				$month_num = date('m');
		}
		
		$fields = array('Event.id', 'Event.name', 'Event.slug','DAY(start_date) AS event_day');
						
		$conditions = array(
		                    'MONTH(Event.start_date)'=>$month_num,
		                    'YEAR(Event.start_date)' => $year
		                    );
						
		if($city!='all-cities') $conditions['Event.city_id']=$city_id;
						
		$v = $this->Event->find('all',array(
		      'conditions'=>$conditions,
          'fields'=>array('count(Event.id) as c', 'DAY(start_date) as day','start_date'),
          'group'=> 'day'));
        
	  $ev_id = null;
	  foreach($v as $event):
  	  $data[$event[0]['day']] = "<a href=\"/events/index/".$city."/all-tags/".$year."/".$month."/".$event[0]['day']." \" title=\"".$event[0]['c']."\"><img src=\"/img/star.png\" alt=\"Events\"/></a>";
	  endforeach;
		
    if($tag_id) {
      $tag_conditions['conditions']['tag_id'] = $tag_id;
      $tag_conditions['fields'] = 'event_id';      
      $events_id = $this->EventsTag->find('list',$tag_conditions);
    }

    $conditions= null;
		if(isset($day)) {
	    $conditions['YEAR(start_date)']   = $year;
	    $conditions['MONTH(start_date)']  = $month_num;
	    $conditions['DAY(start_date)']    = $day;
		}

    if($city_id) $conditions['Event.city_id'] = $city_id; 
    if($tag_id) $conditions['Event.id'] = $events_id;
    		
		$this->set('year', $year);
		$this->set('month', $month);
		$this->set('base_url', $base_url);
		$this->set('data', $data);
		$this->set('city',$city);
		$this->set('city_name',$city_name);
		$this->set('current_tag',$tag);
		$this->Event->recursive=2;
		$this->paginate['contain'] = array('User','Tag','City'=>array('Country'));
    $this->set('events',$this->paginate('Event',$conditions));
	  $this->set('countries',$this->City->getCities());
    $this->set('toptags',$this->Tag->getTopTags($city_id));
	}

  /**
   * Display an Event with all it's data.
   *
   * @param $slug string
   *
   **/
	
	function view($city='all-cities', $slug) {
		$this->Event->recursive=2;
		$this->Event->contain(array(
		               'Comment'=>array(
		                        'User'=>array(
		                             'fields'=>array('username','slug')
		                         )),
		                        'Attendees',
		                        'User',
		                        'Tag',
		                        'City'=>array('Country')));
		$event = $this->Event->find(array('Event.slug'=>$slug));

    if(!$event) {
      $this->redirect('/');
      return false;
    }

		$this->pageTitle = Configure::read('eventoSettings.appName').' | ';
		$this->pageTitle.= $event['City']['name'].' - '.$event['Event']['name'];
		
		$isAttendee = $this->Event->isAttendee($event['Event']['id'],$this->Auth->user('id'));

		$this->set('event', $event);
		$this->set('city',$city);
		$this->set('map',true);
		$this->set('isAttendee',$isAttendee);
	}
	
  /**
   * Add a new Event. If the Event's city and tags does not exist insert
   * it all in the database.
   **/
	
	function add() {
	  $this->set('countries',$this->Country->getCountriesList());
		if (!empty($this->data)) {
			$clean = new Sanitize();
  		$clean->clean($this->data);
			$this->data['Event']['name']  = $clean->html($this->data['Event']['name'], true);
			$this->data['Event']['notes'] = $clean->html($this->data['Event']['notes'], true);
      $this->data['Event']['user_id'] = $this->Auth->user('id');
      
      $this->data['City']['id'] = $this->City->field('id',array(
                'City.name'=>ucfirst($this->data['City']['name'])));
          
      if($this->Event->saveAll($this->data)) {
        $event_id = $this->Event->getInsertID();
        $event = $this->Event->find(array('Event.id'=>$event_id),
                                    array('Event.slug','City.slug'));
        $tags = explode(',',$this->data['Event']['tags']);
        foreach($tags as $tag):
          $tag = strtolower(trim($tag));
          if($tag!="") {
            $new_tag['Tag']['id']   = $this->Tag->field('id',array('name'=>$tag));
            $new_tag['Tag']['name'] = $tag;
            $new_tag['Event']['id'] = $event_id;
            
            $this->Tag->create();
            $this->Tag->save($new_tag);     
          }
        endforeach;
        $this->flash(__('Event saved', true), '/events/view/'.$event['City']['slug'].'/'.$event['Event']['slug']);
      }
    }
    $this->set('map',true);
	}
   
  /**
   * Edit an existing event.
   *
   * @param $event_id int
   *
   */
  
  function edit($event_id=null) {
    $this->Event->recursive = 0;
    $event=$this->Event->find(array('Event.id'=>$event_id),
            array('Event.id','Event.user_id','Event.slug','City.slug'));

    if(empty($event) or $this->Auth->user('id')!=$event['Event']['user_id']) {
      $this->redirect('/');
      return false;
    }
    if($this->__editEvent($event_id)) {
      $this->flash(__('Event saved', true), '/events/view/'.$event['City']['slug'].'/'.$event['Event']['slug']);
    }
    $this->set('map',true);
  }
     
	/**
	* Function for the Ajax autocompleter used for the city names.
	* It is used in the add form for the Events model.
	**/
	
  function autoComplete() {
    $conditions = array(
            'City.name LIKE' => $this->data['City']['name'].'%'
          );
          
    if(isset($this->data['City']['country_id'])) {
          $conditions['country_id'] = $this->data['City']['country_id'];
    }
    
    $this->set('cities', $this->City->find('all', array(
                                            'conditions' => $conditions,
                                            'fields' => array('name')
                                          )));
    $this->layout = 'ajax';
  }	
  
	/**
	* Function for the Ajax autocompleter used for the tag names.
	* It gets the last tag the user writes and looks for it in database.
	* Used in the add form for the Events model.
	**/
	  
  function autoCompleteTags() {
  
    if(!strrpos($this->data['Event']['tags'],',')) {
      $tag = $this->data['Event']['tags'];
    }
    else {
      $tag = substr($this->data['Event']['tags'],strrpos($this->data['Event']['tags'],',')+1);
      $tag = trim($tag);
      if(strlen($tag)<1) {
        $this->set('tags',array());
        return false;
      }
    }
    
    $conditions = array(
            'Tag.name LIKE' => $tag.'%'
          );

    $this->set('tags', $this->Tag->find('all', array(
                                            'conditions' => $conditions,
                                            'fields' => array('name')
                                          )));
    $this->layout = 'ajax';
  }
  
  /**
   * If something is worng with an event users can report it to the
   * admin via email.
   *
   * @param $event_id int
   *
   */
   
  function report($event_id) {
		$this->pageTitle = Configure::read('eventoSettings.appName').' | '.Configure::read('eventoSettings.appSlogan');	  
    $this->Event->recursive = -1;
    $event = $this->Event->find(array('Event.id'=>$event_id));

    if(!empty($this->data)) {
 
      $this->Email->from    = Configure::read('eventoSettings.systemEmail');
      $this->Email->to      = Configure::read('eventoSettings.adminEmail');
      $this->Email->subject = 'Event Report on '.$event_id.': '.$event['Event']['name'];

      $this->Email->delivery = 'mail';
      $this->Email->send($this->data['Event']['report']);      

      $this->flash(__('Event Reported', true), array('action'=>'index'));
    }
    
    $this->set('event',$event);
  }
  
  /**
   * add attendees to the event.
   */
  
  function addAttendee($event_id) {
    $this->Event->recursive = 0;
    $event = $this->Event->find(array('Event.id'=>$event_id));
    if(empty($event)){ 
      $this->redirect('/');
      return false;
    }
    $this->Event->addAttendee($event_id,$this->Auth->user('id'));
    $this->flash(__('Data Saved',true),'/events/view/'.$event['City']['slug'].'/'.$event['Event']['slug']);           
  }

  /**
   * del attendee to the event.
   */
   
  function delAttendee($event_id) {
    $this->Event->recursive = 0;
    $event = $this->Event->find(array('Event.id'=>$event_id));
    if(empty($event)) $this->redirect('/');

    $this->Event->delAttendee($event_id,$this->Auth->user('id')); 
    $this->flash(__('Data Saved',true),'/events/view/'.$event['City']['slug'].'/'.$event['Event']['slug']);           
  }
  
  //////////////////////////////////////////////////////////////////////
  //////////////////////////////////////////////////////////////////////
  //////////////////////////////////////////////////////////////// ADMIN
  //////////////////////////////////////////////////////////////////////
  //////////////////////////////////////////////////////////////////////
 
  /**
   * Show main admin page with a list of events.
   **/
   
  function admin_index() {
    $events = $this->paginate('Event');
    $this->set('events',$events);
  }

  /**
   * Delete event using its id.
   *
   * @para id int
   *
   **/
  
  function admin_delete($id) {
    $this->Event->delete($id);
    $this->flash(__('Event Deleted', true), array('action'=>'list'));
  }

  /**
   * Delete tags not associated to events.
   **/

  function admin_cleantags() {
    $this->Tag->deleteEmptyTags();
    $this->flash(__('Empty tags deleted', true), array('action'=>'list'));
  }
  
  /**
   * Delete cities without associated to events.
   **/  
  
  function admin_cleancities() {
    $this->City->deleteEmptyCities();
    $this->flash(__('Empty cities deleted',true), array('action'=>'list'));
  }

  /**
   * Edit an existing event.
   *
   * @param $event_id int
   *
   */
  
  function admin_edit($event_id) {
    if($this->__editEvent($event_id)) {
        $this->flash(__('Event saved', true), array('action'=>'index'));    
    }
    else {
      $this->set('map',true);
      $this->render('edit');
    }
  }
  
  //////////////////////////////////////////////////////////////////////
  ///////////////////// Private Methods
  
  function __editEvent($event_id) {
    if(!empty($this->data)) {
			$clean = new Sanitize();
  		$clean->clean($this->data);
  		
  		$this->data['Event']['id'] = $event_id;
			$this->data['Event']['name']  = $clean->html($this->data['Event']['name'], true);
			$this->data['Event']['notes'] = $clean->html($this->data['Event']['notes'], true);
      
      $this->data['City']['id'] = $this->City->field('id',array(
                'City.name'=>strtolower($this->data['City']['name'])));
          
      if($this->Event->saveAll($this->data)) {
  
        $this->Tag->deleteEventTags($event_id);
        
        $tags = explode(',',$this->data['Event']['tags']);
        foreach($tags as $tag):
          $tag = strtolower(trim($tag));
          if($tag!="") {
            $new_tag['Tag']['id']   = $this->Tag->field('id',array('name'=>$tag));
            $new_tag['Tag']['name'] = $tag;
            $new_tag['Event']['id'] = $event_id;
            
            $this->Tag->create();
            $this->Tag->save($new_tag);     
          }
        endforeach;
        return true;
      }
    }
    $this->set('countries',$this->Country->getCountriesList());
    $this->Event->recursive = 1;
    $this->data=$this->Event->find(array('Event.id'=>$event_id));
    $tag_list = "";
    foreach($this->data['Tag'] as $tag):
      if(!empty($tag_list)) $tag_list .= ', ';
      $tag_list.= $tag['name'];
    endforeach;		
		$this->data['Event']['tags'] = $tag_list;
  }
}
?>