<?php
/**
* Filename.......: class.smtp.inc
* Project........: SMTP Class
* Version........: 1.0.5
* Last Modified..: 21 December 2001
*/

	define('SMTP_STATUS_NOT_CONNECTED', 1, true);
	define('SMTP_STATUS_CONNECTED', 2, true);

	class smtp{

		var $authenticated;
		var $connection;
		var $recipients;
		var $headers;
		var $timeout;
		var $errors;
		var $status;
		var $body;
		var $from;
		var $host;
		var $port;
		var $helo;
		var $auth;
		var $user;
		var $pass;

		/**
        * Constructor function. Arguments:
		* $params - An assoc array of parameters:
		*
		*   host    - The hostname of the smtp server		Default: localhost
		*   port    - The port the smtp server runs on		Default: 25
		*   helo    - What to send as the HELO command		Default: localhost
		*             (typically the hostname of the
		*             machine this script runs on)
		*   auth    - Whether to use basic authentication	Default: false
		*   user    - Username for authentication			Default: <blank>
		*   pass    - Password for authentication			Default: <blank>
		*   timeout - The timeout in seconds for the call	Default: 5
		*             to fsockopen()
        */

		function smtp($params = array()){

			if(!defined('CRLF'))
				define('CRLF', "\r\n", true);

			$this->authenticated	= false;
			$this->timeout			= 5;
			$this->status			= SMTP_STATUS_NOT_CONNECTED;
			$this->host				= 'localhost';
			$this->port				= 25;
			$this->helo				= 'localhost';
			$this->auth				= false;
			$this->user				= '';
			$this->pass				= '';
			$this->errors   		= array();

			foreach($params as $key => $value){
				$this->$key = $value;
			}
		}

		/**
        * Connect function. This will, when called
		* statically, create a new smtp object, 
		* call the connect function (ie this function)
		* and return it. When not called statically,
		* it will connect to the server and send
		* the HELO command.
        */

		function &connect($params = array()){

			if ($this==null || !isset($this->status)) {
				$obj = new smtp($params);
				if($obj->connect()) {
					$obj->status = SMTP_STATUS_CONNECTED;
				}

				return $obj;

			}
			$this->connection = fsockopen($this->host, $this->port, $errno, $errstr, $this->timeout);
			if (function_exists('socket_set_timeout')) {
				@socket_set_timeout($this->connection, 5, 0);
			}

			$greeting = $this->get_data();
			if (is_resource($this->connection)){
				return $this->auth ? $this->ehlo() : $this->helo();
			}
			$this->errors[] = 'Failed to connect to server: '.$errstr;
			return false;
		}

		/**
        * Function which handles sending the mail.
		* Arguments:
		* $params	- Optional assoc array of parameters.
		*            Can contain:
		*              recipients - Indexed array of recipients
		*              from       - The from address. (used in MAIL FROM:),
		*                           this will be the return path
		*              headers    - Indexed array of headers, one header per array entry
		*              body       - The body of the email
		*            It can also contain any of the parameters from the connect()
		*            function
        */

		function send($params = array()) {

			foreach($params as $key => $value){
				$this->set($key, $value);
			}

			// Abort immediately if not connected
			if (!$this->is_connected()) {
				$this->errors[] = 'Not connected!';
				return false;
			}


            // Start SMTP session


			// Do we auth or not? Note the distinction between the auth variable and auth() function
			if ($this->auth && !$this->authenticated) {
				if (!$this->auth()) {
					return false;
				}
			}

			// Set from address (FROM part)
			if (!$this->mail($this->from)) {
				return false;
			}

			// Set recipient or recipients (RCPT part)
			if (is_array($this->recipients)) {
				foreach($this->recipients as $value) {
					if (!$this->rcpt($value)) {
						return false;
					}
				}
			} else {
				if (!$this->rcpt($this->recipients)) {
					return false;
				}
			}

			// Extra headers and contents (DATA part)
			if (!$this->data()) {
             return false;
			}


			// Transparency
			$headers = str_replace(CRLF.'.', CRLF.'..', trim(implode(CRLF, $this->headers)));
			$body    = str_replace(CRLF.'.', CRLF.'..', $this->body);
			$body    = $body[0] == '.' ? '.'.$body : $body;

			$this->send_data($headers);
			$this->send_data('');
			$this->send_data($body);
			$this->send_data('.');

			$result = (substr(trim($this->get_data()), 0, 3) === '250');
			//$this->rset();
			return $result;
		}
		
		/**
        * Function to implement HELO cmd
        */

		function helo() {
			if (is_resource($this->connection)
					&& $this->send_data('HELO '.$this->helo)
					&& substr(trim($error = $this->get_data()), 0, 3) === '250' ) {

				return true;

			}
			$this->errors[] = 'HELO command failed, output: ' . trim(substr(trim($error),3));
			return false;
		}
		
		/**
        * Function to implement EHLO cmd
        */

		function ehlo() {
			if (is_resource($this->connection)
					&& $this->send_data('EHLO '.$this->helo)
					&& substr(trim($error = $this->get_data()), 0, 3) === '250' ) {

				return true;

			}
			$this->errors[] = 'EHLO command failed, output: ' . trim(substr(trim($error),3));
			return false;
		}
		
		/**
        * Function to implement RSET cmd
        */

		function rset() {
			if (is_resource($this->connection)
					&& $this->send_data('RSET')
					&& substr(trim($error = $this->get_data()), 0, 3) === '250' ) {

				return true;

			}
			$this->errors[] = 'RSET command failed, output: ' . trim(substr(trim($error),3));
			return false;
		}
		
		/**
        * Function to implement QUIT cmd
        */

		function quit() {
			if (is_resource($this->connection)
					&& $this->send_data('QUIT')
					&& substr(trim($error = $this->get_data()), 0, 3) === '221' ) {

				fclose($this->connection);
				$this->status = SMTP_STATUS_NOT_CONNECTED;
				return true;

			}
			$this->errors[] = 'QUIT command failed, output: ' . trim(substr(trim($error),3));
			return false;
		}
		
		/**
        * Function to implement AUTH cmd
        */

		function auth(){
			if (is_resource($this->connection)
					&& $this->send_data('AUTH LOGIN')
					&& substr(trim($error = $this->get_data()), 0, 3) === '334'
					&& $this->send_data(base64_encode($this->user))			// Send username
					&& substr(trim($error = $this->get_data()),0,3) === '334'
					&& $this->send_data(base64_encode($this->pass))			// Send password
					&& substr(trim($error = $this->get_data()),0,3) === '235' ) {

				$this->authenticated = true;
				return true;

			}
			$this->errors[] = 'AUTH command failed: ' . trim(substr(trim($error),3));
			return false;
		}

		/**
        * Function that handles the MAIL FROM: cmd
        */
		
		function mail($from) {

			$error = "";
			if ($this->is_connected()
					&& $this->send_data('MAIL FROM:<'.$from.'>')
					&& substr(trim($error = $this->get_data()), 0, 3) === '250' ) {

				return true;

			}
			$this->errors[] = trim(substr(trim($error), 3));
			return false;
		}

		/**
        * Function that handles the RCPT TO: cmd
        */
		
		function rcpt($to) {

			$error = "";
			if ($this->is_connected()
					&& $this->send_data('RCPT TO:<'.$to.'>')
					&& substr(trim($error = $this->get_data()), 0, 2) === '25' ) {

				return true;

			}
			$this->errors[] = trim(substr(trim($error), 3));
			return false;
		}

		/**
        * Function that sends the DATA cmd
        */

		function data(){

			$error = "";
			if ($this->is_connected()
					&& $this->send_data('DATA')
					&& substr(trim($error = $this->get_data()), 0, 3) === '354' ) {

				return true;
			}
			$this->errors[] = trim(substr(trim($error), 3));
            return false;
		}

		/**
        * Function to determine if this object
		* is connected to the server or not.
        */

		function is_connected() {

			return (is_resource($this->connection) && ($this->status === SMTP_STATUS_CONNECTED));
		}

		/**
        * Function to send a bit of data
        */

		function send_data($data){

			if (is_resource($this->connection)) {
				return fwrite($this->connection, $data.CRLF, strlen($data) + strlen(CRLF));
			}
			return false;
		}

		/**
        * Function to get data.
        */

		function &get_data(){

			$return = '';
			$line   = '';
			$loops  = 0;

			if(is_resource($this->connection)){
				while((strpos($return, CRLF) === false || substr($line,3,1) !== ' ')
						&& $loops < 100) {

					$line    = fgets($this->connection, 512);
					$return .= $line;
					$loops++;
				}
				return $return;

			}
			return false;
		}

		/**
        * Sets a variable
        */
		
		function set($var, $value){

			$this->$var = $value;
			return true;
		}

	} // End of class
?>