<?php
/**
* Filename.......: addressbook.php
* Project........: V-webmail
* Last Modified..: $Date: 2006/03/04 12:55:13 $
* CVS Revision...: $Revision: 1.5 $
* Copyright......: 2001-2004 Richard Heyes
*/

/**
* Common addressbook functionality
*/

class addressbook
{
	/**
    * Addresses
	* @var array
    */
	var $addresses;
	
	/**
    * Groups
	* @var array
    */
	var $groups;
	
	/**
    * Categories list
	* @var array
    */
	var $categories;

	/**
    * Constructor Loads the addressbook data
    */
	function addressbook()
	{
		$this->addresses = array();
		$this->groups    = array();
		$this->uid       = 0;

		$addressbook = loadAddressBook();

		$this->addresses  = @$addressbook['addresses'];
		$this->groups     = @$addressbook['groups'];
		$this->categories = @$addressbook['categories'];
		$this->uid        = @$addressbook['uid'];
	}

	/**
    * Saves the addressbook data
    */
	function save()
	{
		return saveAddressbook(array('addresses' => $this->addresses, 'groups' => $this->groups, 'categories' => $this->categories, 'uid' => $this->uid));
	}
	
	/**
    * Returns a serialized version of the addresses for use
	* in the export to v-webmail user
	*
	* @param array $uids Optional list of uids to 
    */
	function getSerializedExport($uids = null)
	{
		if (!empty($uids) AND is_array($uids)) {
			foreach ($this->addresses as $address) {
				if (in_array($address['uid'], $uids)) {
					$addresses[] = $address;
				}
			}
		} else {
			$addresses = $this->addresses;
		}

		return serialize($addresses);
	}
	
	/**
    * Sorts the addressbook
	*
	* @param string  $sorton  Criteria to sort by, surname, forename, or email
	* @param integer $ascdesc 1 == Ascending, 0 == descending
    */
	function sort($sorton, $ascdesc)
	{
		$args_order = $ascdesc ? sprintf('$a[\'%1$s\'], $b[\'%1$s\']', $sorton) : sprintf('$b[\'%1$s\'], $a[\'%1$s\']', $sorton);
		usort($this->addresses, create_function('$a, $b', sprintf('return @strcmp(%s);', $args_order)));
	}

	/**
    * Returns an address
	*
	* @param integer $uid Unique ID of the array of the address desired
    */
	function getAddress($uid)
	{
		$index = $this->getArrayIndexFromUid($uid);
		return $this->addresses[$index];
	}

	/**
    * Returns all addresses
    */
	function getAllAddresses($nameOnly = false, $selectedUID = 0)
	{
		$return = array();
		foreach ($this->addresses as $address) {
			$selected = $address['uid'] == $selectedUID ? 'selected="selected"' : '';
			$return[] = array('uid' => $address['uid'], 'text' => $this->createStringFromAddress($address, $nameOnly), 'selected' => $selected);
		}

		return $return;
	}

	/**
    * Returns the master list of categories
	*
	* @return array The list of categories
    */
	function getCategoryList()
	{
		return (array)$this->categories;
	}

	/**
    * Updates the defined category list
	* 
	* @param array $categories The categories to merge with the master list
    */
	function updateCategories($categories = array())
	{
		$this->categories = array_unique(array_merge((array)$this->categories, $categories));
		
		// Ensure all categories are specified somewhere
		$validCategories = array();
		foreach ($this->categories as $key => $category) {
			foreach ($this->addresses as $address) {
				if (@in_array($category, $address['categories'])) {
					$validCategories[] = $category;
					continue 2;
				}
			}
		}

		$this->categories = $validCategories;
	}

	/**
    * Sets an addresses property
	*
	* @param integer $uid      The unique id of the address
	* @param string  $property The property to set
	* @param string  $value    The value to set
    */
	function setAddressProperty($uid, $property, $value)
	{
		$index = $this->getArrayIndexFromUid($uid);
		$this->addresses[$index][$property] = $value;
	}

	/**
    * Adds an address to the addressbook
	*
	* @param string $address The address to add
    */
	function addAddress($address)
	{
		global $CONFIG;
		include_once($CONFIG['pear_dir'] . 'Mail/RFC822.php');
		include_once($CONFIG['pear_dir'] . 'Net/URL.php');

		$url = &new Net_URL();
		$address = Mail_RFC822::parseAddressList($address, $url->host, null, false);

		if (!PEAR::isError($address)) {
			if (!empty($address[0]->personal)) {
	
				$personal = preg_replace('/^("|\')(.*)(\1)$/', '\2', $address[0]->personal); // Remove any surrounding quotes "/'
				$personal = preg_split('/\s+/', $personal, -1, PREG_SPLIT_NO_EMPTY);
	
				switch (true) {
					case count($personal) == 1:
						$fname = $personal[0];
						$lname = '';
						break;
		
					case count($personal) == 2:
						$fname = $personal[0];
						$lname = $personal[1];
						break;
		
					case count($personal) > 2:
						$fname = $personal[0];
						array_shift($personal);
						$lname = implode(' ', $personal);
						break;
				}
			} else {
				$fname = '';
				$lname = '';
			}
	
			$this->addresses[] = array('forename' => $fname,
			                           'surname'  => $lname,
								       'email'    => $address[0]->mailbox . '@' . $address[0]->host,
									   'uid'      => $this->uid++
			                          );
		}
	}
	
	/**
    * Adds an address directly, only updating the UID
	*
	* @param array $address The address data
    */
	function importAddress($address)
	{
		$address['uid'] = $this->uid++;
		$this->addresses[] = $address;

		// Update category list.
		if (!empty($address['categories'])) {
			$this->updateCategories($address['categories']);
		}

		return $address['uid'];
	}

	/**
    * Deletes an address (not forgetting group membership)
    */
	function deleteAddress($uid)
	{
		$index = $this->getArrayIndexFromUid($uid);
		if (isset($this->addresses[$index])) {
			unset($this->addresses[$index]);
		}
		if (is_array($this->groups)) {
			// Delete from any groups the address may be a member of
			foreach ($this->groups as $group_name => $group_members) {
				if (in_array($uid, $group_members)) {
					for ($i=0; $i<count($this->groups[$group_name]); $i++) {
						if ($this->groups[$group_name][$i] == $uid) {
							unset($this->groups[$group_name][$i]);
						}
					}
					$this->groups[$group_name] = array_values($this->groups[$group_name]);
				}
			}
		}
	}

	/**
    * Deletes a group
	*
	* @param string $group Name of group to delete
    */
	function deleteGroup($group)
	{
		if (isset($this->groups[$group])) {
			unset($this->groups[$group]);
		}
	}

	/**
    * Creates a new group
	*
	* @param string $name The groupname
    */
	function createGroup($name)
	{
		$this->groups[$name] = array();
	}
	
	/**
    * Adds an address to a group from uid
	*
	* @param integer $uid UID to add
    */
	function addAddressToGroup($group, $uid)
	{
		if (!in_array($uid, (array)$this->groups[$group])) {
			$this->groups[$group][] = $uid;
		}
	}

	/**
    * Returns true/false if a uid is in a group
	*
	* @param integer $uid  UID to check
	* @param string $group Group name
    */
	function inGroup($uid, $group)
	{
		return in_array($uid, $this->groups[$group]);
	}

	/**
    * Returns list of members for a group
	*
	* @param string $group Name of group
    */
	function getGroupMembers($group)
	{
		$return = array();
		if (!empty($this->groups[$group])) {
			foreach ($this->groups[$group] as $uid) {
				$return[] = array('uid' => $uid, 'text' => $this->createStringFromUid($uid));
			}
		} elseif ($group == lang('All addresses')) {
			foreach (@(array)$this->addresses as $address) {
				$return[] = array('uid' => $address['uid'], 'text' => $this->createStringFromAddress($address));
			}
		}

		return $return;
	}
	
	/**
    * Sets the group members of a group
	*
	* @param string $group   Name of the group
	* @param array  $members Array of address uids
    */
	function setGroupMembers($group, $members)
	{
		if (!$this->groups[$group]) {
			$this->createGroup($group);
		}
		$this->groups[$group] = $members;
	}

	/**
    * Returns a list of groups
    */
	function getGroupList($withAll = true)
	{
		$return = array();

		if ($withAll) {
			$all_members = array();
			// Add "All addresses" group first
			foreach ($this->addresses as $address) {
				$all_members[] = $this->createLinkStringFromAddress($address);
			}
			$all_members = array('name' => lang('All addresses'), 'members' => implode(', ', $all_members));
		}

		foreach ((array)$this->groups as $group_name => $group_members) {
			$members = array();
			foreach ((array)$group_members as $uid) {
				$members[] = $this->createLinkStringFromUid($uid);
			}
			$return[] = array('name' => $group_name, 'members' => implode(', ', $members));
		}

		// Sort, and then prepend the "All addresses" group
		usort($return, create_function('$a, $b', 'return strcmp($a["name"], $b["name"]);'));
		if ($withAll) {
			array_unshift($return, $all_members);
		}
		
		return $return;
	}

	/**
    * Returns full name given the fore/surname values
	*
	* @param  array $data The data to work from
	* @return string      The full name
    */
	function getFullName($data)
	{
		return trim(@$data['forename'] . ' ' . @$data['surname']);
	}
	
	/**
    * Returns full name given the UID
	*
	* @param  integer $uid The UID to work from
	* @return string       The full name
    */
	function getFullNameByUid($uid)
	{
		if (($index = $this->getArrayIndexFromUid($uid)) !== false) {
			return $this->getFullName($this->addresses[$index]);
		}
		
		return false;
	}

	/**
    * Creates a string from addressbook info
	*
	* @param array $data The data to work from
    */
	function createStringFromAddress($data, $nameOnly = false)
	{
		$return = addressbook::getFullName($data);

		// Need quotes ?
		if ('' != trim($return)) {
	        if ((!preg_match('/^[\\x00-\\x7E]+$/i', $return, $matches) OR
			     preg_match('/[][()<>@,;\\:".]/', $return) OR
				 preg_match('/[\\x00-\\x1F]+/', $return)) AND
				 !$nameOnly) {
				$return = sprintf('"%s"', $return);
			}
		}
		
		if (!empty($data['email']) AND !$nameOnly) {
			$return .= sprintf(' <%s>', $data['email']);
		}
		
		return $return;
	}

	/**
    * Creates an address string given a uid
	*
	* @param integer $uid UID of address
    */
	function createStringFromUid($uid)
	{
		$index = $this->getArrayIndexFromUid($uid);
		return $this->createStringFromAddress($this->addresses[$index]);
	}

	/**
    * As above but creates also a link to the compose
	* page
	*
	* @param array $data The data to work from
    */
	function createLinkStringFromAddress($data)
	{
		$address = addressbook::createStringFromAddress($data);
		
		return sprintf('<a href="%s" title="%s %s">%s</a>', common::url('email.compose.php?to=' . urlencode($address)), lang('Send mail to'), javascript::escapeString(htmlspecialchars($data['email'])), htmlspecialchars($address));
	}
	
	/**
    * As above but takes uid instead of address data
	*
	* @param integer $uid The uid to work with
    */
	function createLinkStringFromUid($uid)
	{
		$index = $this->getArrayIndexFromUid($uid);
		return $this->createLinkStringFromAddress($this->addresses[$index]);
	}
	
	/**
    * Return array index of the address in the the
	* member "addresses" based on the uid
	*
	* @param integer $uid The uid to work with
    */
	function getArrayIndexFromUid($uid)
	{
		foreach ($this->addresses as $id => $address) {
			if ($address['uid'] == $uid) {
				return $id;
			}
		}

		return false;
	}
	
	/**
    * Creates a vCard from an addressbook entry
	*
	* @param  integer $uid The UID of the addressbook entry
	* @return string       The raw vCard data
    */
	function createVcardFromUID($uid)
	{
		if ( ($index = $this->getArrayIndexFromUid($uid)) !== false) {
			return $this->createVcardFromAddress($this->addresses[$index]);
		}
		
		return false;
	}
	
	/**
    * Creates a vCard from the given address data. Can be
	* called statically.
	*
	* @param  array $data The address data
	* @return string      The raw vCard data
    */
	function createVcardFromAddress($data)
	{
		require_once($GLOBALS['CONFIG']['includes'] . 'vCard.php');

		$vcf = new vCard();
		$vcf->addName(@$data['surname'], @$data['forename'], '', @$data['title']);
		$vcf->addFormattedName(@$data['surname'] . ' ' . @$data['forename']);

		// Optional fields
		if (!empty($data['street']) OR !empty($data['city']) OR !empty($data['province']) OR !empty($data['postcode']) OR !empty($data['country'])) {
			$vcf->addAddress($data['street'], $data['city'], $data['province'], $data['postcode'], $data['country']);
		}
		
		
		// Telephone numbers
		if (!empty($data['telephone_home'])) {
			$vcf->addTelephone($data['telephone_home'], array('HOME', 'VOICE'));
		}
		if (!empty($data['telephone_work'])) {
			$vcf->addTelephone($data['telephone_work'], array('WORK', 'VOICE'));
		}
		if (!empty($data['telephone_cell'])) {
			$vcf->addTelephone($data['telephone_cell'], array('CELL', 'VOICE'));
		}
		if (!empty($data['telephone_fax'])) {
			$vcf->addTelephone($data['telephone_fax'], array('FAX'));
		}

		// Default email
		if (!empty($data['email'])) {
			$vcf->addEmail($data['email'], array('INTERNET', 'PREF'));
		}

		// Other emails
		if (!empty($data['email_2'])) {
			$vcf->addEmail($data['email_2'], array('INTERNET'));
		}

		if (!empty($data['email_3'])) {
			$vcf->addEmail($data['email_3'], array('INTERNET'));
		}
		
		// Notes
		if (!empty($data['notes'])) {
			$vcf->addNote($data['notes']);
		}
		
		// Categories
		if (!empty($data['categories'])) {
			$vcf->addCategories($data['categories']);
		}

		return $vcf->generateVcard();
	}
}
?>