﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Text.RegularExpressions;
using System.IO;

namespace CssSpriteGenerator
{
    public class Validators
    {
        /// <summary>
        /// Returns true if the given string s does not have characters that are illegal in
        /// CSS identifiers, such as class names.
        /// 
        /// Note that this method does not check whether the string is a valid identifier,
        /// because it doesn't check the start of the string.
        /// 
        /// According to http://www.w3.org/TR/CSS21/syndata.html:
        /// In CSS, identifiers (including element names, classes, and IDs in selectors) can contain only 
        /// the characters [a-zA-Z0-9] and ISO 10646 characters U+00A0 and higher, plus the hyphen (-) and 
        /// the underscore (_); they cannot start with a digit, two hyphens, or a hyphen followed by a digit. 
        /// Identifiers can also contain escaped characters and any ISO 10646 character as a numeric code (see next item). 
        /// For instance, the identifier "B&W?" may be written as "B\&W\?" or "B\26 W\3F".
        /// 
        /// </summary>
        /// <param name="s"></param>
        /// <returns></returns>
        public static bool IsValidCssIdentifierComponent(string s)
        {
            bool hasWrongCharacter = Regex.IsMatch(s, @"[^a-zA-Z0-9_\-]");
            return !hasWrongCharacter;
        }

        /// <summary>
        /// Returns true if the given string is both a valid folder name in a url and in
        /// the file system.
        /// 
        /// For characters allowed in file system folders, see
        /// http://msdn.microsoft.com/en-us/library/aa365247(v=vs.85).aspx
        /// 
        /// For characters allowed in URIs, see
        /// http://labs.apache.org/webarch/uri/rfc/rfc3986.html
        /// </summary>
        /// <param name="s"></param>
        /// <returns></returns>
        public static bool IsValidFolderName(string s)
        {
            // Ensure that the string only has characters from a safe character set:
            // letters, digits, underscore, hyphen, period, percent (useful for character encoding) and plus (for encoding space).
            // Backward slash (to create subfolders) is also ok.
            // By allowing letters, we also allow letters from non-English languages.

            string goodCharacters = @"\w\-\.%+\\";
            string badCharactersRegexp = "[^" + goodCharacters + "]";
            bool hasWrongCharacter = Regex.IsMatch(s, badCharactersRegexp);
            if (hasWrongCharacter) { return false; }

            // Path can't end in .
            if (s.EndsWith(".")) { return false; }

            // Don't use reserved names
            // The folder cannot be called "CON", "PRN", etc. and also not "Con.txt", etc.

            string reservedNamesRegexp = 
                @"^(?:CON|PRN|AUX|NUL|COM1|COM2|COM3|COM4|COM5|COM6|COM7|COM8|COM9|LPT1|LPT2|LPT3|LPT4|LPT5|LPT6|LPT7|LPT8|LPT9|\.\.|\.)(?:\..*)$";
            bool hasReservedName = Regex.IsMatch(s, reservedNamesRegexp, RegexOptions.IgnoreCase);
            if (hasReservedName) { return false; }

            return true;
        }

    }
}
