﻿#region SVN File History
// ***********************************************************
// Copyright ©   2010 Craig Greenock.
// Contact     - cgreenock@bcs.org.uk
// Version 0.0 - 05 September 2010 19:40
// Machine     - KEFALLONIA Microsoft Windows NT 6.1.7600.0
//
// Implement dialogues as hidden DIVs (cf context menus).
// An abstract base class that provides the default event 
// handling for derived classes.
//
// $Author: cig $
// $HeadURL: https://kefallonia/svn/projects_2008/LurkerWeb/trunk/LurkerWeb/CustomControl/DialogueBase.cs $
// $LastChangedBy: cig $
// $LastChangedDate: 2011-03-13 11:12:25 +0000 (Sun, 13 Mar 2011) $
// $Revision: 252 $
//
// ***********************************************************
#endregion  // SVN File History

#region References
using System;
using System.Collections.Specialized;
using System.Web.UI;
using System.Web.UI.WebControls;
using System.Web.UI.HtmlControls;
using System.Drawing;
using System.Reflection;
using System.IO;

#endregion // References

namespace Demo
{
  public abstract class DialogueBase : WebControl
  {

    #region Constants

    #region DialogueHelper javascript methods
    private const string sectionBreak = "<hr style='height:1px;border:solid 1px black;' />";
    private const string attachDialogue =  "return __showDialogue('{0}')";
    private const string trapEscKey = "__trapESC({0})";
    #endregion // DialogueHelper javascript methods

    private const string hideOnClick = "{0}.style.display = 'none';";
    private const string onMouseOver = "this.style.background = '{0}';";
    private const string onMouseOut = "this.style.background = '{0}';";

    private const string DialogueTag = "dialogue";
    private const string UnadornedName = "unadornedname";

    /// <summary>
    /// Use the old inband file, group and record separator 
    /// characters to identify constituent parts of a return value.
    /// </summary>
    private enum Delimiter
    {
      FS = 28,
      GS = 29,
      RS = 30,
      US = 31
    }

    #endregion // constants

    #region declarations

    protected Panel childControls;                      // All dialogue controls are displayed on a panel.
    private string dialogueTitle;
    private string dialogueID;                          // A unique identifier for the dialogue.
    private WebControl host;                            // The control that is to request the dialogue.
    public event CommandEventHandler DialogueClose;

    #endregion // declarations

    #region private methods

    /// <summary>
    /// Build an invisible div containing a table of controls.
    /// Dialogues are generally going to be fairly simple so we don't mind waiting 
    /// for them to be rendered and a table makes it easy to align items.
    /// </summary>
    private HtmlGenericControl buildDialogue()
    {

			// A dialogue is an invisible DIV that is displayed 
      // using scripting when the user right-clicks on a bound HTML tag
			HtmlGenericControl div = new HtmlGenericControl("div");
			div.ID = Guid.NewGuid().ToString("N");
			div.Style["Display"] = "none";
			div.Style["position"] = "absolute";

      // Provide somewhere for the derived class to place its controls...
			childControls = new Panel();
      childControls.Width = this.Width;
      childControls.Height = this.Height;
			childControls.ApplyStyle(CreateControlStyle());
			div.Controls.Add(childControls);

      // ...stick on a default header
      this.AddDialogueTitle(childControls);

      // ...and derived class adds all the controls it needs...
      this.AddDialogueControls(childControls);

      // ...and then each input control is marked as owned by "this" dialogue so
      // so we can find them again easily and fish the values out of them prior
      // to postback.
      foreach (object ctl in childControls.Controls)
      {
        try
        {
          // If the control type isn't on this list we ignore it.
          // It's either for layout or not supported.
          if (ctl.GetType() == typeof(TextBox) ||
              ctl.GetType() == typeof(CheckBox) ||
              ctl.GetType() == typeof(RadioButtonList) ||
              ctl.GetType() == typeof(DropDownList))
            {
            WebControl c = (WebControl)ctl;
            c.Attributes.Add(DialogueTag, this.DialogueKey);
            c.Attributes.Add(UnadornedName, c.ID);
            }
        }
        catch {/*NOP*/}
      }


      // ...except the OK and Cancel buttons.
      this.AddDialogueButtons(childControls);

      return div;

    }

    /// <summary>
    /// Push out the javascript necessary to deal with dialogue events
    /// and link the host control to the javascript that invokes the dialogue.
    /// </summary>
    private void linkToRequester()
    {
      writeScriptBlock();
      host.Attributes.Remove("onclick");
      host.Attributes.Add("onclick", getDialogueReference());
    }

    /// <summary>
    /// Embed javascript dialogue functions if not yet embedded.
    /// </summary>
    private void writeScriptBlock()
    {
      if (!Page.ClientScript.IsStartupScriptRegistered("DialogueHelper"))
      {
        string js = readResourceString("DialogueHelper.js");
        Page.ClientScript.RegisterClientScriptBlock(typeof(String), "DialogueHelper", js);
      }
    }

    /// <summary>
    /// Load resource from manifest. Remember to set your build action to 'embed resource'
    /// for the resource file.
    /// </summary>
    /// <param name="resourceName"></param>
    /// <returns></returns>
    private string readResourceString(string resourceName)
    {
      Assembly dll = Assembly.GetExecutingAssembly();
      StreamReader reader;
      reader = new StreamReader(dll.GetManifestResourceStream(resourceName.ManifestResourcePath()));
      string script = reader.ReadToEnd();
      reader.Close();

      return script;
    }

    /// <summary>
    /// Generate javascript necessary to hook an html element (control) to javascript.
    /// </summary>
    /// <returns></returns>
    private string getDialogueReference()
    {
      return String.Format(attachDialogue, Controls[0].ClientID);
    }

    /// <summary>
    /// Generate javascript to dismiss menu on ESC.
    /// </summary>
    /// <returns></returns>
    private string getEscReference()
    {
      return String.Format(trapEscKey, Controls[0].ClientID);
    }

    #endregion // private methods

    #region protected methods & overrides

    #region OK/Cancel button click handling

    /// <summary>
    /// Retrieve the etx separated string name value pairs from the 
    /// HTTPRequest, convert to name value collection and fire the 
    /// single external click event.
    /// </summary>
    /// <param name="sender"></param>
    /// <param name="e"></param>
    protected virtual void onButtonClick(object sender, EventArgs e)
    {
      ContextMenuButton button = (ContextMenuButton)sender;
      if (button != null)
      {

        NameValueCollection returnData = new NameValueCollection();

        if (CommandCancel == button.CommandArgument.ToUpper())
          returnData.Add(CommandCancel, true.ToString());
        else
        {
          returnData = DialogueData(Page.Request);
        }        

        CommandEventArgs args = new CommandEventArgs(button.CommandArgument, 
                                                     returnData);
        OnDialogueClose(args);
      }
    }

    /// <summary>
    /// Fire the dialogue closed event.
    /// </summary>
    /// <param name="e"></param>
    protected virtual void OnDialogueClose(CommandEventArgs e)
    {
      if (DialogueClose != null)
        DialogueClose(this, e);
    }

    #endregion  // OK/Cancel button click handling

    #region control creation

    /// <summary>
    /// Provide default means to put a title on the dialogue.
    /// </summary>
    /// <param name="controlContainer"></param>
    /// <returns></returns>
    protected virtual Panel AddDialogueTitle(Panel controlPanel)
    {

      Label title = new Label();
      title.Text = DialogueTitle;
      title.Font.Underline = true;
      title.Font.Bold = true;
      controlPanel.Controls.Add(title);

      Literal line = new Literal();
      line.Text = "<hr/>";
      controlPanel.Controls.Add(line);
      
      return controlPanel;
    }

    /// <summary>
    /// Provide default buttons auto wired to click handler.
    /// </summary>
    /// <param name="buttonContainer"></param>
    /// <returns></returns>
    protected virtual Panel AddDialogueButtons(Panel buttonContainer)
    {

      // Put the buttons in a "bar" at the bottom and to the
      // right of the panel.
      Literal line = new Literal();
      line.Text = "<hr/>";
      buttonContainer.Controls.Add(line);

      // Yes. I know we should use CSS, but this is easier.
      Table t = new Table();
      TableRow r = new TableRow();
      TableCell c = new TableCell();

      t.Rows.Add(r);
      t.HorizontalAlign = HorizontalAlign.Right;
      t.CellPadding = 5;

      ContextMenuButton OK = new ContextMenuButton();
      OK.Text = "OK";
      OK.CommandArgument = DialogueBase.CommandOK;
      OK.CommandName = DialogueBase.CommandOK;
      OK.CustomPostback = CustomOK;
      OK.Click += onButtonClick;
      
      OK.Width = Unit.Pixel(50);
      c.HorizontalAlign = HorizontalAlign.Right;
      c.Controls.Add(OK);
      r.Cells.Add(c);

      ContextMenuButton Cancel = new ContextMenuButton();
      Cancel.Text = "Cancel";
      Cancel.CommandArgument = DialogueBase.CommandCancel;
      Cancel.CommandName = DialogueBase.CommandCancel;
      Cancel.CustomPostback = CustomCancel;
      Cancel.Click += onButtonClick;
      Cancel.Width = Unit.Pixel(50);
      c = new TableCell();
      c.HorizontalAlign = HorizontalAlign.Right;
      r.Cells.Add(c);
      c.Controls.Add(Cancel);

      buttonContainer.Controls.Add(t);
      return buttonContainer;

    }

    /// <summary>
    /// Provide an alternative postback to the base _
    /// postback. This allows us to extract the data the user enters in the
    /// dialogue.
    /// </summary>
    /// <param name="owner">Control that fires the postback.</param>
    /// <returns>String representation of a javascript funct</returns>
    protected string CustomOK(ContextMenuButton owner) 
    {
      return string.Format("__getValues('{0}', '{1}', '{2}', '{3}')", 
                                        owner.UniqueID, 
                                        DialogueTag, 
                                        this.dialogueID, 
                                        UnadornedName);;
    }

    protected string CustomCancel(ContextMenuButton owner)
    {
      return string.Format("__cancelDialogue('{0}', '{1}', '{2}')", 
                                            owner.UniqueID, 
                                            this.dialogueID,
                                            owner.CommandArgument);
    }

    /// <summary>
    /// Build the dialogue when requested and establish the link
    /// with the control that will request the dialogue.
    /// </summary>
    protected override void CreateChildControls()
    {
      Controls.Add(buildDialogue());
      linkToRequester();
    }

    /// <summary>
    /// Decide what the dialogue is going to look like (very plain as it happens).
    /// </summary>
    /// <returns></returns>
    protected override Style CreateControlStyle()
    {
      Style style = base.CreateControlStyle();
      style.BorderStyle = BorderStyle.Outset;
      style.BorderColor = Color.DarkGray;
      style.BorderWidth = Unit.Pixel(2);
      style.BackColor = Color.LightGray;
      style.ForeColor = this.ForeColour;
      style.Font.Name = "verdana";
      style.Font.Size = FontUnit.Point(10);
      return style;
    }

    /// <summary>
    /// Derived class implements this method to place required controls.
    /// </summary>
    /// <param name="controlContainer">A panel to hold the derived classes controls.</param>
    /// <returns>The amended panel</returns>
    protected abstract Panel AddDialogueControls(Panel controlContainer);

    #endregion // control creation

    #endregion // protected methods & overrides

    #region public methods

    #region State checking methods

    /// <summary>
    /// A "constant" for checking what sort of command is returned.
    /// </summary>
    protected static string CommandOK
    {
      get {return "OK";}
    }

    /// <summary>
    /// A "constant" for checking what sort of command is returned.
    /// </summary>
    protected static string CommandCancel
    {
      get {return "CANCEL";}
    }

    /// <summary>
    /// Allow a client programmer to check whether the postback is 
    /// from an identified dialogue.
    /// </summary>
    /// <param name="DialogueKey">The constant ID assigned to the ID in its construction.</param>
    /// <param name="Request">An HTTPRequest.</param>
    /// <returns></returns>
    public static Boolean IsDialogue(string DialogueKey, System.Web.HttpRequest Request)
    {
      char gs = (char)Delimiter.GS;
      PostbackInfo pi = PostbackInfo.Make(Request, gs.ToString());
      return (pi.Argument.Length > 1 &&
              pi.Argument[0].ToUpper() == DialogueKey.ToUpper());
    }

    /// <summary>
    /// Take an HTTPRequest and extract the dialogue field data as
    /// a name value collection.
    /// </summary>
    /// <param name="Request"></param>
    /// <returns></returns>
    public static NameValueCollection DialogueData(System.Web.HttpRequest Request)
    {

      // DialogueKey{GS}name=value{RS}name=value{RS}....
      // It's assumed that a check that the Request contains
      char gs = (char)Delimiter.GS;  
      PostbackInfo pi = PostbackInfo.Make(Request, gs.ToString());

      NameValueCollection returnData = new NameValueCollection();

      // If we have a header and at least on name value pair process
      // the argument.
      if (pi.Argument.Length > 1)
      {
        char rs = (char)Delimiter.RS;
        string[] values = pi.Argument[1].Split(rs);
        foreach(string returnPair in values)
        {
            returnData.Add(returnPair.Split('=')[0],
                           returnPair.Split('=')[1]);
        } 
      }

      return returnData;

    }

    /// <summary>
    /// Determine if Dialogue is cancelled directly from HTTPRequest 
    /// </summary>
    /// <param name="Request">HTTPRequest containing postback data.</param>
    /// <returns></returns>
    public static bool IsCancelled(System.Web.HttpRequest Request)
    {
      return IsCancelled(DialogueData(Request));
    }

    /// <summary>
    /// Does dialogue data indicate cancellation?
    /// </summary>
    /// <param name="dialogueData">Name value collection of data extract from HTTPRequest</param>
    /// <returns></returns>
    public static bool IsCancelled(NameValueCollection dialogueData)
    {

      bool cancelled = false;
      return (dialogueData[DialogueBase.CommandCancel] != null &&
              Boolean.TryParse(dialogueData[DialogueBase.CommandCancel], out cancelled) &&
              cancelled);
    }

    #endregion // State checking methods.

    #region Basic Appearance

    /// <summary>
    /// Sometimes a title might be nice.
    /// </summary>
    public string DialogueTitle
    {
      get { return dialogueTitle; }
      set { dialogueTitle = value; }
    }

    /// <summary>
    /// Set a fixed width for the menu.
    /// </summary>
    public override Unit Width
    {
      get 
      {
        object width = ViewState["dialogueWidth"];
        if (width==null) width = Unit.Percentage(100);
        return (Unit)width;
      }
      set {ViewState["dialogueWidth"] = value;}
    }

    /// <summary>
    /// Set a fixed height for the dialogue.
    /// </summary>
    public override Unit Height
    {
      get 
      {
        object height = ViewState["dialogueHeight"];
        if (height==null) height = Unit.Percentage(100);
        return (Unit)height;
      }
      set {ViewState["dialogueHeight"] = value;}
    }

    /// <summary>
    /// Set text colour.
    /// </summary>
    public Color ForeColour
    {
      get 
      {
        object colour = ViewState["foreColour"];
        if (colour==null) colour = Color.Black;
        return (Color)colour;
      }
      set {ViewState["foreColour"] = value;}
    }

    /// <summary>
    /// Number of blank pixels around each control item
    /// </summary>
    public int CellPadding
    {
      get 
      {
        object padding = ViewState["CellPadding"];
        if (padding == null)
          padding = 2;
        return (int) padding;
      }
      set 
      {
        ViewState["CellPadding"] = value;
      }
    }

    #endregion // Basic Appearance

    #region Control Binding

    /// <summary>
    /// Bind the dialogue to a single web control such as a button.
    /// </summary>
    /// <param name="BindTo"></param>
    public WebControl BoundControl
    {
      get{return host;}
      set{host = value;}
    }


    /// <summary>
    /// A unique identifier for this dialogue instance. Used when retrieving
    /// the dialogue's values. It must be a constant or at the very least obtained
    /// in a deterministic manner. 
    /// </summary>
    public string DialogueKey
    {
      get { return dialogueID;}
      set { dialogueID = value;}
    }

    #endregion // Control Binding

    #endregion // public methods

  }
}
