﻿using System.Diagnostics;
using System.IO;
using System.Text;
using System.Web.Script.Serialization;
using System.Xml.Serialization;
using System.Linq;

namespace System.Web.Mvc
{
    public class SerializedDataResult : ActionResult
    {
        public SerializedDataResult()
        {
            SerializedDataFormat = SerializedDataFormat.Auto;
            SerializedDataRequestBehavior = SerializedDataRequestBehavior.DenyGet;
        }

        public Encoding ContentEncoding { get; set; }

        public string ContentType { get; set; }

        public object Data { get; set; }

        public SerializedDataFormat SerializedDataFormat { get; set; }

        public string FormatParameter { get; set; }

        public SerializedDataRequestBehavior SerializedDataRequestBehavior { get; set; }

        public override void ExecuteResult(ControllerContext context)
        {
            if (context == null)
            {
                throw new ArgumentNullException("context");
            }
            if (SerializedDataRequestBehavior == SerializedDataRequestBehavior.DenyGet &&
                String.Equals(context.HttpContext.Request.HttpMethod, "GET", StringComparison.OrdinalIgnoreCase))
            {
                throw new InvalidOperationException("This request has been blocked because sensitive information could be disclosed to third party web sites when this is used in a GET request. To allow GET requests, set SerializedDataRequestBehavior to AllowGet.");
            }

            HttpResponseBase response = context.HttpContext.Response;

            if (ContentEncoding != null)
            {
                response.ContentEncoding = ContentEncoding;
            }

            // try to detect requested format from request headers
            if (SerializedDataFormat == SerializedDataFormat.Auto)
                SerializedDataFormat = DetectRequestedSerializedDataFormat(context.HttpContext.Request);

            switch (SerializedDataFormat)
            {
                case SerializedDataFormat.Xml:
                    SerializeAsXml(response);
                    break;
                default:
                    SerializeAsJson(response);
                    break;
            }
        }

        private SerializedDataFormat DetectRequestedSerializedDataFormat(HttpRequestBase request)
        {
            SerializedDataFormat format = SerializedDataFormat.Auto;

            // look for an explicit request param
            if (String.IsNullOrEmpty(FormatParameter))
                FormatParameter = "format";

            if (!String.IsNullOrEmpty(request[FormatParameter]))
            {
                SerializedDataFormat result;
                if (Enum.TryParse(request[FormatParameter], true, out result))
                    format = result;
            }

            // look for recognized types in the accept-types header
            if (format == SerializedDataFormat.Auto)
            {
                if (request.AcceptTypes != null && request.AcceptTypes.Contains("text/xml"))
                    format = SerializedDataFormat.Xml;
                else if (request.AcceptTypes != null && request.AcceptTypes.Contains("application/json"))
                    format = SerializedDataFormat.Json;
            }

            // fall back to Json if no type detected
            if (format == SerializedDataFormat.Auto)
                format = SerializedDataFormat.Json;

            return format;
        }

        private void SerializeAsXml(HttpResponseBase response)
        {
            response.ContentType = !String.IsNullOrEmpty(ContentType) ? ContentType : "text/xml";

            if (Data != null)
            {
                XmlSerializer serializer = new XmlSerializer(Data.GetType());
                MemoryStream stream = new MemoryStream();
                StreamWriter writer = ContentEncoding == null
                                          ? new StreamWriter(stream) {AutoFlush = true}
                                          : new StreamWriter(stream, ContentEncoding) {AutoFlush = true};
                serializer.Serialize(writer, Data);
                stream.Position = 0;
                StreamReader reader = new StreamReader(stream);
                response.Write(reader.ReadToEnd());
            }
        }

        private void SerializeAsJson(HttpResponseBase response)
        {
            response.ContentType = !String.IsNullOrEmpty(ContentType) ? ContentType : "application/json";

            if (Data != null)
            {
                JavaScriptSerializer serializer = new JavaScriptSerializer();
                response.Write(serializer.Serialize(Data));
            }
        }
    }

    public enum SerializedDataFormat
    {
        Auto,
        Json,
        Xml,
    }

    public enum SerializedDataRequestBehavior
    {
        AllowGet,
        DenyGet,
    }
}