﻿using System;
using System.Text;
using System.Web.Mvc;
using Microsoft.VisualStudio.TestTools.UnitTesting;
using Moq;

namespace SerializedDataResultTests
{
    [TestClass]
    public class SerializedDataResultTest
    {
        private object _data = new Quote()
                                   {
                                       Ask = 20,
                                       Bid = 10,
                                       Code = "ABC",
                                       LastTradePrice = 15,
                                       LastTradeTime = DateTime.Parse("2011-01-01"),
                                       Volume = 10000
                                   };
        private string _jsonSerializedData =
            "{\"Code\":\"ABC\",\"LastTradePrice\":15,\"LastTradeTime\":\"\\/Date(1293800400000)\\/\"," +
            "\"Bid\":10,\"Ask\":20,\"Volume\":10000}";

        private string _xmlSerializedData =
            @"<?xml version=""1.0"" encoding=""utf-8""?>
<Quote xmlns:xsi=""http://www.w3.org/2001/XMLSchema-instance"" xmlns:xsd=""http://www.w3.org/2001/XMLSchema"">
  <Code>ABC</Code>
  <LastTradePrice>15</LastTradePrice>
  <LastTradeTime>2011-01-01T00:00:00</LastTradeTime>
  <Bid>10</Bid>
  <Ask>20</Ask>
  <Volume>10000</Volume>
</Quote>";

        [TestMethod]
        public void ExplicitJsonResult()
        {
            // Arrange
            object data = _data;
            Encoding contentEncoding = Encoding.UTF8;

            // Arrange expectations
            Mock<ControllerContext> mockControllerContext = new Mock<ControllerContext>(MockBehavior.Strict);
            mockControllerContext.SetupGet(c => c.HttpContext.Request.HttpMethod).Returns("POST").Verifiable();
            mockControllerContext.SetupSet(c => c.HttpContext.Response.ContentType = "application/json").Verifiable();
            mockControllerContext.SetupSet(c => c.HttpContext.Response.ContentEncoding = contentEncoding).Verifiable();
            mockControllerContext.Setup(c => c.HttpContext.Response.Write(_jsonSerializedData)).Verifiable();

            SerializedDataResult result = new SerializedDataResult()
            {
                Data = data,
                SerializedDataFormat = SerializedDataFormat.Json,
                ContentType = String.Empty,
                ContentEncoding = contentEncoding
            };

            // Act
            result.ExecuteResult(mockControllerContext.Object);

            // Assert
            mockControllerContext.Verify();
        }

        [TestMethod]
        public void ExplicitXmlResult()
        {
            // Arrange
            object data = _data;
            Encoding contentEncoding = Encoding.UTF8;

            // Arrange expectations
            Mock<ControllerContext> mockControllerContext = new Mock<ControllerContext>(MockBehavior.Strict);
            mockControllerContext.SetupGet(c => c.HttpContext.Request.HttpMethod).Returns("POST").Verifiable();
            mockControllerContext.SetupSet(c => c.HttpContext.Response.ContentType = "text/xml").Verifiable();
            mockControllerContext.SetupSet(c => c.HttpContext.Response.ContentEncoding = contentEncoding).Verifiable();
            mockControllerContext.Setup(c => c.HttpContext.Response.Write(_xmlSerializedData)).Verifiable();

            SerializedDataResult result = new SerializedDataResult()
            {
                Data = data,
                SerializedDataFormat = SerializedDataFormat.Xml,
                ContentType = String.Empty,
                ContentEncoding = contentEncoding
            };

            // Act
            result.ExecuteResult(mockControllerContext.Object);

            // Assert
            mockControllerContext.Verify();
        }

        [TestMethod]
        public void GetRequestBlocked()
        {
            // Arrange expectations
            Mock<ControllerContext> mockControllerContext = new Mock<ControllerContext>(MockBehavior.Strict);
            mockControllerContext.SetupGet(c => c.HttpContext.Request.HttpMethod).Returns("GET").Verifiable();

            SerializedDataResult result = new SerializedDataResult();

            // Act & Assert
            try
            {
                result.ExecuteResult(mockControllerContext.Object);
                Assert.Fail("InvalidOperationException expected.");
            }
            catch (InvalidOperationException ex)
            {
                Assert.AreEqual(ex.Message, "This request has been blocked because sensitive information could be disclosed to third party web sites when this is used in a GET request. To allow GET requests, set SerializedDataRequestBehavior to AllowGet.");
            }
            catch
            {
                Assert.Fail("InvalidOperationException expected.");
            }

            mockControllerContext.Verify();
        }

        [TestMethod]
        public void AutoFormatUsesRequestInput()
        {
            // Arrange
            object data = _data;

            // Arrange expectations
            Mock<ControllerContext> mockControllerContext = new Mock<ControllerContext>(MockBehavior.Strict);
            mockControllerContext.SetupGet(c => c.HttpContext.Request.HttpMethod).Returns("POST").Verifiable();
            mockControllerContext.SetupGet(c => c.HttpContext.Request["format"]).Returns("xml").Verifiable();
            mockControllerContext.SetupSet(c => c.HttpContext.Response.ContentType = "text/xml").Verifiable();
            mockControllerContext.Setup(c => c.HttpContext.Response.Write(_xmlSerializedData)).Verifiable();

            SerializedDataResult result = new SerializedDataResult()
            {
                Data = data,
                SerializedDataFormat = SerializedDataFormat.Auto,
            };

            // Act
            result.ExecuteResult(mockControllerContext.Object);

            // Assert
            mockControllerContext.Verify();
        }

        [TestMethod]
        public void AutoFormatUsesAcceptTypes()
        {
            // Arrange
            object data = _data;

            // Arrange expectations
            Mock<ControllerContext> mockControllerContext = new Mock<ControllerContext>(MockBehavior.Strict);
            mockControllerContext.SetupGet(c => c.HttpContext.Request.HttpMethod).Returns("POST").Verifiable();
            mockControllerContext.SetupGet(c => c.HttpContext.Request["format"]).Returns(String.Empty).Verifiable();
            mockControllerContext.SetupGet(c => c.HttpContext.Request.AcceptTypes).Returns(new[] { "text/xml" }).Verifiable();
            mockControllerContext.SetupSet(c => c.HttpContext.Response.ContentType = "text/xml").Verifiable();
            mockControllerContext.Setup(c => c.HttpContext.Response.Write(_xmlSerializedData)).Verifiable();

            SerializedDataResult result = new SerializedDataResult()
            {
                Data = data,
                SerializedDataFormat = SerializedDataFormat.Auto,
            };

            // Act
            result.ExecuteResult(mockControllerContext.Object);

            // Assert
            mockControllerContext.Verify();
        }
    }
}
