#region Copyright(c) Anton Shelin, Vladimir Timashkov. All Rights Reserved.
// -----------------------------------------------------------------------------
// Copyright(c) 2010 Anton Shelin, Vladimir Timashkov. All Rights Reserved.
//
// Redistribution and use in source and binary forms, with or without
// modification, are permitted provided that the following conditions are met:
//
//   1. No Trademark License - Microsoft Public License (Ms-PL) does not grant you rights to use
//      authors names, logos, or trademarks.
//   2. If you distribute any portion of the software, you must retain all copyright,
//      patent, trademark, and attribution notices that are present in the software.
//   3. If you distribute any portion of the software in source code form, you may do
//      so only under this license by including a complete copy of Microsoft Public License (Ms-PL)
//      with your distribution. If you distribute any portion of the software in compiled
//      or object code form, you may only do so under a license that complies with
//      Microsoft Public License (Ms-PL).
//   4. The names of the authors may not be used to endorse or promote products
//      derived from this software without specific prior written permission.
//
// The software is licensed "as-is." You bear the risk of using it. The authors
// give no express warranties, guarantees or conditions. You may have additional consumer
// rights under your local laws which this license cannot change. To the extent permitted
// under your local laws, the authors exclude the implied warranties of merchantability,
// fitness for a particular purpose and non-infringement.
// -----------------------------------------------------------------------------
#endregion

using System;
using System.Collections.Generic;
using System.Reflection;
using NUnit.Framework;

namespace SharpDom.UnitTests
{
    [TestFixture]
    public class TagTests
    {
        private bool _isInitialized = false;
        private readonly List<Type> _tagClasses = new List<Type>();
        private readonly Dictionary<Type, List<PropertyInfo>> _tagAttributes = new Dictionary<Type, List<PropertyInfo>>();

        [SetUp]
        public void Init()
        {
            if (_isInitialized) return;
            var types = typeof(IndexedTag).Assembly.GetTypes();
            foreach (var type in types)
            {
                if (!type.IsSubclassOf(typeof(IndexedTag))) continue;
                if (type == typeof(ContainerTag) || type.IsSubclassOf(typeof(ContainerTag))) continue;
                if (type == typeof(TagCustom)) continue;
                _tagClasses.Add(type);

                var properties =  type.GetProperties(BindingFlags.Public | BindingFlags.Instance);
                var attributes = new List<PropertyInfo>();
                foreach(var property in properties)
                {
                    if (!property.PropertyType.IsSubclassOf(typeof(BaseAttr))) continue;
                    attributes.Add(property);
                }
                _tagAttributes.Add(type, attributes);
            }
            _isInitialized = true;
        }

        [Test]
        public void Test_AttributeClassName_StartsWith_PredefinedPrefix()
        {
            foreach (var tagClass in _tagClasses)
            {
                Assert.That(tagClass.Name.StartsWith(Constants.TagClassPrefix), Is.True,
                    string.Format("Tag class '{0}' doesn't start with prefix '{1}'",
                    tagClass.Name, Constants.TagClassPrefix));
            }
        }

        [Test]
        public void Test_ClassName_IsEqualTo_AttributeName()
        {
            foreach (var tagClass in _tagClasses)
            {
                var constructor = tagClass.GetConstructor(Type.EmptyTypes);
                var tag = (IndexedTag)constructor.Invoke(null);
                var className = tagClass.Name.Substring(Constants.TagClassPrefix.Length);
                var tagName = tag.TagName;

                Assert.That(className.ToLower() == tagName, Is.True,
                    string.Format("Tag class '{0}' has incorrect name of tag '{1}'", className, tagName));
            }
        }

        [Test]
        public void Test_AttributeProperty_Is_Virtual()
        {
            foreach (var tagClass in _tagClasses)
            {
                var attributes = _tagAttributes[tagClass];
                foreach (var attribute in attributes)
                {
                    var getter = attribute.GetGetMethod();
                    var setter = attribute.GetGetMethod();

                    Assert.That(getter.IsVirtual, Is.True, "Tag class '{0}' has not virtual getter");
                    Assert.That(setter.IsVirtual, Is.True, "Tag class '{0}' has not virtual setter");
                }
            }
        }

        [Test]
        public void Test_AttributeName_IsEqualTo_AttributeClassName()
        {
            foreach (var tagClass in _tagClasses)
            {
                var attributes = _tagAttributes[tagClass];
                foreach (var attribute in attributes)
                {
                    var propertyName = attribute.Name;
                    var className = attribute.PropertyType.Name.Substring(Constants.AttributeClassPrefix.Length);

                    Assert.That(propertyName == className, Is.True, "Attribute '{0}' of tag '{1}' has incorrect name '{2}'",
                        attribute.PropertyType.Name, tagClass.Name, attribute.Name);
                }
            }
        }

        [Test]
        public void Test_AttrMethod_Is_PresentAndVirtual()
        {
            foreach (var tagClass in _tagClasses)
            {
                var attrMethod = tagClass.GetMethod(Constants.AttrMethod);
                Assert.That(attrMethod, Is.Not.Null,
                    string.Format("Tag '{0}' doesn't have '{1}' public method", tagClass.Name, Constants.AttrMethod));
                Assert.That(attrMethod.IsVirtual, Is.True,
                    string.Format("'{0}' method of the tag '{1}' is not public", Constants.AttrMethod, tagClass.Name));
            }
        }

        [Test]
        public void Test_Constructor_Has_SimilarSetOfParameters_As_SetOfAttributes()
        {
            foreach (var tagClass in _tagClasses)
            {
                var attributes = _tagAttributes[tagClass];
                var attrMethod = tagClass.GetMethod(Constants.AttrMethod);
                var parameters = new List<ParameterInfo>(attrMethod.GetParameters());

                // check that for each tag attribute there is a parameter in the constructor
                foreach (var attribute in attributes)
                {
                    var foundParameter = parameters.Find(x => x.ParameterType == attribute.PropertyType);
                    Assert.That(foundParameter, Is.Not.Null,
                        string.Format("'{0}' method of the tag '{1}' doesn't have parameter for attribute '{2}'",
                        Constants.AttrMethod, tagClass.Name, attribute.Name));
                }

                // check that for each parameter in the constructor there is tag attribute
                foreach (var parameter in parameters)
                {
                    var foundAttribute = attributes.Find(x => x.PropertyType == parameter.ParameterType);
                    Assert.That(foundAttribute, Is.Not.Null,
                        string.Format("'{0}' method of the tag '{1}' has extra parameter '{2}' which does not correspond to any tag's attribute'",
                        Constants.AttrMethod, tagClass.Name, parameter.Name));
                }
            }
        }

        [Test]
        public void Test_Constructor_Has_ProperlyNamedParameters()
        {
            foreach (var tagClass in _tagClasses)
            {
                var attrMethod = tagClass.GetMethod(Constants.AttrMethod);
                var parameters = new List<ParameterInfo>(attrMethod.GetParameters());
                foreach (var parameter in parameters)
                {
                    var parameterName = parameter.Name;
                    var className = parameter.ParameterType.Name.Substring(Constants.AttributeClassPrefix.Length).ToLower();

                    Assert.That(parameterName == className, Is.True, "Parameter '{0}' of the '{1}' method of the tag '{2}' has incorrect name '{3}'",
                        parameter.ParameterType.Name, Constants.AttrMethod, tagClass.Name, parameter.Name);
                }
            }
        }

        [Test]
        public void Test_Tag_Has_CorrectParentshipInterface()
        {
            foreach (var tagClass in _tagClasses)
            {
                var parentships = tagClass.GetCustomAttributes(typeof (ParentshipAttribute), false);

                Assert.That(parentships.Length, Is.EqualTo(1),
                    string.Format("Tag '{0}' doesn't have parentship interface attached", tagClass.Name));

                var parentshipData = (ParentshipAttribute) parentships[0];
                var interfaceName = parentshipData.ParentshipInterface.Name;
                var tagName = Constants.ParentshipInterface(tagClass.Name);

                Assert.That(interfaceName == tagName, Is.True, "Parentship interface of the tag '{0}' has incorrect value '{1}'",
                    tagClass.Name, parentshipData.ParentshipInterface.Name);
            }
        }

        [Test]
        public void Test_Tag_Has_CorrectAdoptabilityInterface()
        {
            foreach (var tagClass in _tagClasses)
            {
                var adoptabilities = tagClass.GetCustomAttributes(typeof(AdoptabilityAttribute), false);

                Assert.That(adoptabilities.Length, Is.EqualTo(1),
                    string.Format("Tag '{0}' doesn't have adoptability interface attached", tagClass.Name));

                var adoptabilityData = (AdoptabilityAttribute)adoptabilities[0];
                var interfaceName = adoptabilityData.AdoptabilityInterface.Name;
                var tagName = Constants.AdoptabilityInterface(tagClass.Name);

                Assert.That(interfaceName == tagName, Is.True, "Adoptability interface of the tag '{0}' has incorrect value '{1}'",
                    tagClass.Name, adoptabilityData.AdoptabilityInterface.Name);
            }
        }
    }
}