#region Copyright(c) Anton Shelin, Vladimir Timashkov. All Rights Reserved.
// -----------------------------------------------------------------------------
// Copyright(c) 2010 Anton Shelin, Vladimir Timashkov. All Rights Reserved.
//
// Redistribution and use in source and binary forms, with or without
// modification, are permitted provided that the following conditions are met:
//
//   1. No Trademark License - Microsoft Public License (Ms-PL) does not grant you rights to use
//      authors names, logos, or trademarks.
//   2. If you distribute any portion of the software, you must retain all copyright,
//      patent, trademark, and attribution notices that are present in the software.
//   3. If you distribute any portion of the software in source code form, you may do
//      so only under this license by including a complete copy of Microsoft Public License (Ms-PL)
//      with your distribution. If you distribute any portion of the software in compiled
//      or object code form, you may only do so under a license that complies with
//      Microsoft Public License (Ms-PL).
//   4. The names of the authors may not be used to endorse or promote products
//      derived from this software without specific prior written permission.
//
// The software is licensed "as-is." You bear the risk of using it. The authors
// give no express warranties, guarantees or conditions. You may have additional consumer
// rights under your local laws which this license cannot change. To the extent permitted
// under your local laws, the authors exclude the implied warranties of merchantability,
// fitness for a particular purpose and non-infringement.
// -----------------------------------------------------------------------------
#endregion

using System;
using System.IO;
using SharpDom.Navigation;

namespace SharpDom
{
    /// <summary>It is a container tag keeping nested tags. Generated HTML
    /// will show only nested tags not the container tag itself.</summary>
    [Adoptability(typeof(IFullAdoptability))]
    public abstract class ContainerTag : IndexedTag
    {
        public ContainerTag() : base("container") { }

        /// <summary>Indexer accepting an array of nested tags</summary>
        /// <param name="items">Child tags</param>
        /// <returns>This instance</returns>
        public new ContainerTag this[params TreeLeafTag[] items]
        {
            get { return (ContainerTag)base[items]; }
        }

        #region Render

        /// <summary>Determines accumulated indent size of the current tag on rendering stage</summary>
        public override int? IndentSize
        {
            get { return (Parent != null ? Parent.IndentSize : null); }
        }

        /// <summary>Takes part in HTML generation</summary>
        /// <param name="writer">Destionation of rendering</param>
        public override void Render(TextWriter writer)
        {
            if (!IsVisible) return;

            FireOnPreRender();
            Children.Render(writer);
            FireOnPostRender();
        }

        #endregion
    }

    /// <summary>It is a operator tag keeping tags generated for this operator.</summary>
    [Adoptability(typeof(IFullAdoptability))]
    public class OperatorTag<TViewModel> : ContainerTag
    {
        /// <summary>Keeps reference to partial builder building this boundary</summary>
        public virtual TViewModel ViewModel { get; protected set; }

        /// <summary>Default constructor assigning the view model</summary>
        public OperatorTag(TViewModel viewModel)
        {
            ViewModel = viewModel;
        }
    }

    /// <summary>It is a placeholder tag keeping tags generated inside placeholder.</summary>
    [Adoptability(typeof(IFullAdoptability))]
    public class PlaceHolderTag : ContainerTag
    {
        /// <summary>Keeps reference to place holder delegate generating nested tags</summary>
        public virtual PlaceHolderDelegate Layout { get; protected set; }
        /// <summary>Keeps reference to place holder delegate generating nested tags</summary>
        public virtual Func<ContainerTag, ContainerTag> DefaultLayout { get; protected set; }

        /// <summary>Default constructor</summary>
        public PlaceHolderTag(PlaceHolderDelegate layout, Func<ContainerTag, ContainerTag> defaultLayout)
        {
            Layout = layout;
            DefaultLayout = defaultLayout;
        }
    }

    /// <summary>It is a builder tag containing nested tags. Generated HTML
    /// will show only nested tags not the boundary tag itself.</summary>
    [Adoptability(typeof(IFullAdoptability))]
    public class BuilderTag<TViewModel> : ContainerTag
    {
        /// <summary>Keeps reference to partial builder building this boundary</summary>
        public virtual UserControlBuilder<TViewModel> Builder { get; protected set; }

        /// <summary>Default constructor assigning the builder</summary>
        public BuilderTag(UserControlBuilder<TViewModel> builder)
        {
            Builder = builder;
        }
    }
}