#region Copyright(c) Anton Shelin, Vladimir Timashkov. All Rights Reserved.
// -----------------------------------------------------------------------------
// Copyright(c) 2010 Anton Shelin, Vladimir Timashkov. All Rights Reserved.
//
// Redistribution and use in source and binary forms, with or without
// modification, are permitted provided that the following conditions are met:
//
//   1. No Trademark License - Microsoft Public License (Ms-PL) does not grant you rights to use
//      authors names, logos, or trademarks.
//   2. If you distribute any portion of the software, you must retain all copyright,
//      patent, trademark, and attribution notices that are present in the software.
//   3. If you distribute any portion of the software in source code form, you may do
//      so only under this license by including a complete copy of Microsoft Public License (Ms-PL)
//      with your distribution. If you distribute any portion of the software in compiled
//      or object code form, you may only do so under a license that complies with
//      Microsoft Public License (Ms-PL).
//   4. The names of the authors may not be used to endorse or promote products
//      derived from this software without specific prior written permission.
//
// The software is licensed "as-is." You bear the risk of using it. The authors
// give no express warranties, guarantees or conditions. You may have additional consumer
// rights under your local laws which this license cannot change. To the extent permitted
// under your local laws, the authors exclude the implied warranties of merchantability,
// fitness for a particular purpose and non-infringement.
// -----------------------------------------------------------------------------
#endregion

using System;
using System.IO;
using SharpDom;

namespace ConsoleSample
{
    /// <summary>
    /// This sample demonstrates how to design and to use templated user controls.
    /// The difference between standard user controls and templated user controls is that
    /// templated user control delegates creating some pieces of its HTML layout to owner
    /// of this templated user controls. 
    /// In other words, we reuse the master-page-approach but in reverse direction.
    /// The nearest analogy in ASP.NET is "TemplateItem" in Repeator control.
    /// 1. During designing the user control, you should specify placeholder delegates
    /// and additionally you could supply default HTML layouts for the case when the owner
    /// of the user control doesn't specify its version of the content for the placeholders.
    /// 2. In the page builder, you attach your new HTML layouts to user controls' placeholders.
    /// In this case, you can use page builder's view model not the user control's view model.
    /// </summary>
    public class Sample6TemplatedControls
    {
        public static void Run()
        {
            // initially we prepare view model
            var model = new ViewModel
            {
                Title = "Page Title",
                ControlModel = new ControlModel
                {
                    Text = "Some text inside the control",
                    Value = "value of the control"
                }
            };

            // then we create page builder
            var pageBuilder = new PageBuilder();
            var writer = new StringWriter();

            // then we supply the page builder with the model
            pageBuilder.Render(model, writer);

            // finally we show generated HTML code
            Console.WriteLine(writer.ToString());
            Console.ReadKey();
        }

        // ----- Model and Builder -----

        /// <summary>The view model is the same</summary>
        public class ViewModel
        {
            public string Title { get; set; }
            public ControlModel ControlModel { get; set; }
        }

        /// <summary>Now HTML layout is not only partial, it is double partial.
        /// Page builder delegates to create rather big piece of HTML layout to the user control,
        /// and the user control in turn delegates to the page builder to create small very specific
        /// islands of HTML layouts in reverse.</summary>
        public class PageBuilder : HtmlPageBuilder<ViewModel>
        {
            /// <summary>Instance of the user control - should be public and virtual.</summary>
            public virtual ControlBuilder ControlBuilder { get { return CreateBuilder<ControlBuilder>(); } }

            /// <summary>Again, again and again - this methods should be overriden in all page builders.
            /// It should contain real HTML layout expressed in terms of C# language.</summary>
            /// <param name="container">The container inside which new HTML layout should be placed.</param>
            /// <returns>The same container tag as incoming one</returns>
            public override ContainerTag CreateTagTree(ContainerTag container)
            {
                return container
                [
                    html[
                        head[
                            title[Model.Title]
                        ],
                        body[
                            div[

                                // SEE HERE ! - example of attaching to user controls' placeholder
                                ControlBuilder.Attach<ControlBuilder>(b => b.SomePlaceHolder = c => c[

                                    // SEE HERE ! - here specific content of user control's placeholder goes
                                    span[ "Specific content from page " + Model.Title ]

                                ]).CreateTags(Model.ControlModel)
                            ]
                        ]
                    ]
                ];
            }
        }

        // ----- Model and Builder of the User Control -----

        public class ControlModel
        {
            public string Text { get; set; }
            public string Value { get; set; }
        }

        public class ControlBuilder : UserControlBuilder<ControlModel>
        {
            /// <summary>SEE HERE ! - placeholder which the page builder will attach to in order to supply its content</summary>
            public virtual PlaceHolderDelegate SomePlaceHolder { get; set; }

            /// <summary>Again - this methods should be overriden not only in page builders
            /// but also in control builders as well. It should contain real HTML layout expressed
            /// in terms of C# language.</summary>
            /// <param name="container">The container inside which new HTML layout should be placed.</param>
            /// <returns>The same container tag as incoming one</returns>
            public override ContainerTag CreateTagTree(ContainerTag container)
            {
                return container
                [
                    PlaceHolder(SomePlaceHolder, c => c[

                        // SEE HERE ! - default content rendered if the owner doesn't supply specific substitution
                        span[ "Default content is " + Model.Text ]

                    ]),
                    div[ Model.Value ]
                ];
            }
        }
    }
}