#region Copyright(c) Anton Shelin, Vladimir Timashkov. All Rights Reserved.
// -----------------------------------------------------------------------------
// Copyright(c) 2010 Anton Shelin, Vladimir Timashkov. All Rights Reserved.
//
// Redistribution and use in source and binary forms, with or without
// modification, are permitted provided that the following conditions are met:
//
//   1. No Trademark License - Microsoft Public License (Ms-PL) does not grant you rights to use
//      authors names, logos, or trademarks.
//   2. If you distribute any portion of the software, you must retain all copyright,
//      patent, trademark, and attribution notices that are present in the software.
//   3. If you distribute any portion of the software in source code form, you may do
//      so only under this license by including a complete copy of Microsoft Public License (Ms-PL)
//      with your distribution. If you distribute any portion of the software in compiled
//      or object code form, you may only do so under a license that complies with
//      Microsoft Public License (Ms-PL).
//   4. The names of the authors may not be used to endorse or promote products
//      derived from this software without specific prior written permission.
//
// The software is licensed "as-is." You bear the risk of using it. The authors
// give no express warranties, guarantees or conditions. You may have additional consumer
// rights under your local laws which this license cannot change. To the extent permitted
// under your local laws, the authors exclude the implied warranties of merchantability,
// fitness for a particular purpose and non-infringement.
// -----------------------------------------------------------------------------
#endregion

using System;
using System.IO;
using SharpDom;

namespace ConsoleSample
{
    /// <summary>
    /// This sample demonstrates how to design and to use master page approach.
    /// The approach is quite similar to ASP.NET WebForms' Master Pages - 
    /// you design a separated so-called "master page builder". This master page builder
    /// is to contain some placeholders. The placeholders are to be substituted in 
    /// the standard page builders. The chain of {master page, contain page} could be
    /// extended and master page could have super master page and so on.
    /// 1. In master page, you should declare some placeholder delegates. Then in
    /// HTML layout, you can use these delegates with or without default predefined
    /// content for such placeholders.
    /// 2. In content page builder, you should specify functions (with signature
    /// the same as placeholder delegates) which are to supply specific contents.
    /// 3. In content page, in CreateTagsTree method you have to execute 3 steps:
    ///   - create needed master page (so you can choose master page dynamically)
    ///   - attach your functions creating contents to master page's placeholder delegates
    ///   - ask the master page to create combined content.
    /// </summary>
    public class Sample5MasterPages
    {
        public static void Run()
        {
            // initially we prepare view model
            // notice - we create not only view model for content page
            // but also a view model for master page
            var model = new ContentModel
            {
                Text = "Content Text",
                Value = "Content Value",
                MasterModel = new MasterModel
                {
                    Title = "Page Title"
                }
            };

            // then we create page builder
            var pageBuilder = new ContentBuilder();
            var writer = new StringWriter();

            // then we supply the page builder with the model
            pageBuilder.Render(model, writer);

            // finally we show generated HTML code
            Console.WriteLine(writer.ToString());
            Console.ReadKey();
        }

        // ----- Model and Builder of the Master Page -----

        /// <summary>Standard view model - this time it is master page's view model</summary>
        public class MasterModel
        {
            public string Title { get; set; }
        }

        /// <summary>Now HTML layout is not only partial, it is partial in reverse manner now.
        /// And this page is the "schema" of the resulting page</summary>
        public class MasterBuilder : HtmlPageBuilder<MasterModel>
        {
            /// <summary>SEE HERE ! - we declare two placeholder for the master page</summary>
            public virtual PlaceHolderDelegate FirstPlaceHolder { get; set; }
            public virtual PlaceHolderDelegate SecondPlaceHolder { get; set; }

            /// <summary>Again, again and again - this methods should be overriden in all page builders.
            /// It should contain real HTML layout expressed in terms of C# language.</summary>
            /// <param name="container">The container inside which new HTML layout should be placed.</param>
            /// <returns>The same container tag as incoming one</returns>
            public override ContainerTag CreateTagTree(ContainerTag container)
            {
                return container
                [
                    html[
                        head[
                            title[Model.Title]
                        ],
                        body[
                            div[
                                // SEE HERE! - example of placeholder without default content
                                PlaceHolder(FirstPlaceHolder),

                                span[ "some intermediary text" ],

                                // SEE HERE! - example of placeholder with default content
                                PlaceHolder(SecondPlaceHolder, c => c[
                                    span[ "stub for second placeholder" ]
                                ])
                            ]
                        ]
                    ]
                ];
            }
        }

        // ----- Model and Builder of the Content Page -----

        /// <summary>View model of content page - it also includes the model for master page</summary>
        public class ContentModel
        {
            public string Text { get; set; }
            public string Value { get; set; }
            public MasterModel MasterModel;
        }

        /// <summary>Page builder for content page - it defines contents for master page's placeholders</summary>
        public class ContentBuilder : HtmlPageBuilder<ContentModel>
        {
            // SEE HERE! - example of method defining the content for the placeholder
            public virtual ContainerTag FirstPlaceHolderContent(ContainerTag container)
            {
                return container
                [
                    span[ Model.Text ],
                    span[ "somtehing other" ],
                    span[ Model.Value ]
                ];
            }

            // SEE HERE! - example of creating master page and attaching specifc content to it's placeholders
            public override ContainerTag CreateTagTree(ContainerTag container)
            {
                return

                    // SEE HERE ! - example how to create master page
                    CreateBuilder<MasterBuilder>(container)

                    // SEE HERE ! - example how to attach specifc content to master page's placeholders
                    .Attach<MasterBuilder>(b => b.FirstPlaceHolder = FirstPlaceHolderContent)

                    // just call master page method to generate combined content 
                    .CreateTags(Model.MasterModel);
            }
        }
    }
}