﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.IO;
using System.Runtime.Serialization;
using System.Runtime.Serialization.Formatters.Binary;
using System.Security.Principal;
using System.Web.Security;
using System.Configuration;
using System.Web.Hosting;

namespace System.Web
{
    public partial class LocalBase
    {
        public struct ServerCookieData
        {
            public DateTime? ExpiryDate;
            public string Data;
        }

        public static TimeSpan DefaultServerCookieLifeTime = TimeSpan.FromDays(100);

        //Use the following sample to add server cookie configurations to the web.config
        //<configuration>
        //    <appSettings>
        //    <add key="ServerCookieUserKey" value="MyServerCookieUserKey"/>
        //    <add key="ServerCookieFolderPath" value="~/MyServerCookies"/>
        //    </appSettings>
        //</configuration>
        public static string DefaultServerCookieUserNameKey
        {
            get { return ConfigurationManager.AppSettings.Get("ServerCookieUserKey") ?? "ServerCookieUserKey"; }
        }

        public static string DefaultServerCookieFolderPath
        {
            get { return ConfigurationManager.AppSettings.Get("ServerCookieFolderPath") ?? "~\\ServerCookies"; }
        }

        public static void SetServerCookie<T>(bool shared, string key, T value)
        {
            SetServerCookie<T>(null, shared, key, value, false, null, false);
        }

        public static void SetServerCookie<T>(bool shared, string key, T value, bool encrypt)
        {
            SetServerCookie<T>(null, shared, key, value, false, null, encrypt);
        }

        public static void SetServerCookie<T>(bool shared, string key, T value, TimeSpan? lifeTime, bool encrypt)
        {
            SetServerCookie<T>(null, shared, key, value, false, lifeTime, encrypt);
        }

        public static void SetServerCookie<T>(bool shared, string key, T value, bool ignoreMembershipCredentials, TimeSpan? lifeTime, bool encrypt)
        {
            SetServerCookie<T>(null, shared, key, value, ignoreMembershipCredentials, lifeTime, encrypt);
        }

        public static void SetServerCookie<T>(string folderPath, bool shared, string key, T value, bool ignoreMembershipCredentials, TimeSpan? lifeTime, bool encrypt)
        {
            ServerCookieData serverCookieData;
            string userFileName = shared ? key : null;
            string filePath = GetUserFilePath(folderPath, userFileName, ignoreMembershipCredentials);

            if (filePath == null)
                throw new FieldAccessException("Server cookie could not be created under the path " + filePath);

            if (!File.Exists(filePath))
                File.CreateText(filePath).Close();

            try
            {
                Dictionary<string, ServerCookieData> dataDictionary =
                    Local.Serialization.FromJsonString<Dictionary<string, ServerCookieData>>(File.ReadAllText(filePath));

                if (dataDictionary == null)
                    dataDictionary = new Dictionary<string, ServerCookieData>();

                if (value == null)
                {
                    dataDictionary.Remove(key);
                }
                else
                {
                    serverCookieData = new ServerCookieData();
                    if (lifeTime.HasValue)
                        serverCookieData.ExpiryDate = DateTime.Now.Add(lifeTime.Value);

                    serverCookieData.Data = Local.Serialization.ToJsonString<T>(value);

                    if (encrypt)
                        serverCookieData.Data = Local.CookieEncryption.Encrypt(serverCookieData.Data);

                    if (dataDictionary.Keys.Contains(key))
                        dataDictionary[key] = serverCookieData;
                    else
                        dataDictionary.Add(key, serverCookieData);
                }

                using (StreamWriter sw = new StreamWriter(filePath, false))
                {
                    sw.Write(Local.Serialization.ToJsonString<Dictionary<string, ServerCookieData>>(dataDictionary));
                }
            }
            catch (Exception ex)
            {
                throw ex;
            }
        }

        public static T GetServerCookie<T>(bool shared, string key)
        {
            return Local.Serialization.FromJsonString<T>(GetServerCookie(null, shared, key, false, false));
        }

        public static T GetServerCookie<T>(bool shared, string key, bool decrypt)
        {
            return Local.Serialization.FromJsonString<T>(GetServerCookie(null, shared, key, false, decrypt));
        }

        public static T GetServerCookie<T>(bool shared, string key, bool ignoreMembershipCredentials, bool decrypt)
        {
            return Local.Serialization.FromJsonString<T>(GetServerCookie(null, shared, key, ignoreMembershipCredentials, decrypt));
        }

        public static string GetServerCookie(bool shared, string key)
        {
            return GetServerCookie(null, shared, key, false, false);
        }

        public static string GetServerCookie(bool shared, string key, bool decrypt)
        {
            return GetServerCookie(null, shared, key, false, decrypt);
        }

        public static string GetServerCookie(string key, bool ignoreMembershipCredentials, bool decrypt)
        {
            return GetServerCookie(null, false, key, ignoreMembershipCredentials, decrypt);
        }

        public static string GetServerCookie(string folderPath, bool shared, string key, bool ignoreMembershipCredentials, bool decrypt)
        {
            string userFileName = shared ? key : null;
            string filePath = GetUserFilePath(userFileName, ignoreMembershipCredentials);
            if (!File.Exists(filePath))
                return null;

            Dictionary<string, ServerCookieData> dictionary;
            try
            {
                dictionary = Local.Serialization.FromJsonString<Dictionary<string, ServerCookieData>>(File.ReadAllText(filePath));

                if (dictionary != null && dictionary.Keys.Contains(key))
                {
                    if ((dictionary[key].ExpiryDate ?? DateTime.MaxValue) < DateTime.Now)
                        SetServerCookie<object>(shared, key, null);
                    else
                        return decrypt
                            ? Local.CookieEncryption.Decrypt(dictionary[key].Data, true)
                            : dictionary[key].Data;
                }
            }
            catch (Exception ex)
            {
                return null;
            }
            return null;
        }

        public void RemoveServerCookie(bool shared, string key)
        {
            SetServerCookie<object>(shared, key, null);
        }

        public static string GetUserFilePath(bool ignoreMembershipCredentials)
        {
            return GetUserFilePath(null, null, ignoreMembershipCredentials);
        }

        public static string GetUserFilePath(string userFileName, bool ignoreMembershipCredentials)
        {
            return GetUserFilePath(null, userFileName, ignoreMembershipCredentials);
        }

        public static string GetUserFilePath(string folderPath, string fileName, bool ignoreMembershipCredentials)
        {
            if (string.IsNullOrEmpty(fileName))
                fileName = GetUserId(ignoreMembershipCredentials);
            return GetFilePath(folderPath, fileName);
        }

        public static string GetUserId(bool ignoreMembershipCredentials)
        {   //page.Request.LogonUserIdentity
            if (!ignoreMembershipCredentials && !HttpContext.Current.Profile.IsAnonymous)
                return Membership.GetUser(HttpContext.Current.Profile.UserName).ProviderUserKey.ToString();

            string userId = Local.GetCookie(DefaultServerCookieUserNameKey, false, true);
            if (userId != null)
                return userId;
            else // User has no associated record stored on the server.
                return CreateNewUserRecord();
        }

        private static string GetFilePath(string folderPath, string fileName)
        {
            if (string.IsNullOrEmpty(folderPath))
                folderPath = GetServerCookieFolder(DefaultServerCookieFolderPath, true);
            return Path.Combine(folderPath, HttpUtility.UrlEncode(fileName) + ".txt");
        }

        /// <summary>
        /// Create a new text file on the server to store the user data. 
        /// </summary>
        /// <returns>The new unique user Identifier stored in the user cookie.</returns>
        private static string CreateNewUserRecord()
        {
            string userId;
            do
                userId = "User_" + Guid.NewGuid().ToString();
            while (File.Exists(GetFilePath(null, userId))); // Create new user id if it already exists.

            Local.SetCookie(DefaultServerCookieUserNameKey, userId, false, true);
            // Verify that the cookie has been successfully created.
            if (userId == Local.GetCookie(DefaultServerCookieUserNameKey, false, true))
                return userId;
            else
                throw new Exception("Browser does not accept cookies.");
        }

        public static string GetServerCookieFolder(bool createWhenNotFound)
        {
            return GetServerCookieFolder(null, createWhenNotFound);
        }

        public static string GetServerCookieFolder(string folderPath, bool createWhenNotFound)
        {
            if (string.IsNullOrEmpty(folderPath))
                folderPath = DefaultServerCookieFolderPath;

            if (folderPath.StartsWith("~"))
                folderPath = HostingEnvironment.MapPath(folderPath);

            if (createWhenNotFound && !Directory.Exists(folderPath))
                Directory.CreateDirectory(folderPath);

            return folderPath;
        }

        public static long GetServerCookieSize()
        {
            return GetServerCookieSize(false);
        }

        public static long GetServerCookieSize(bool ignoreMembershipCredentials)
        {
            FileInfo file = new FileInfo(GetUserFilePath(ignoreMembershipCredentials));
            if (file.Exists)
                return file.Length;
            else
                return -1;
        }
    }
}
