using System;
using System.Collections.Generic;
using System.Collections.Specialized;
using System.Web;
using System.Collections;
using System.Text;
using System.Web.Security;
using System.Web.Caching;
using System.Web.UI;
using System.Web.SessionState;

#region Storage types
// The storage types from which the value is retrieved.
public enum Sources
{
    None = 0,
    Session = 1,
    Cookie = 2,
    QueryString = 4,
    Cache = 8,
    ServerCookie = 16
    //Memcached = 32
}

// The storage types where the value is saved.
// Each type has is represented by an unsigned integer that can be added up to denote several types within the same integer.
public struct Targets
{
    public static readonly int
    None = 0,
    Session = 1,
    Cookie = 2,
    QueryString = 4,
    Cache = 8,
    ServerCookie = 16;
    //Memcached = 32;
}
#endregion

namespace System.Web
{

    /// <summary>
    /// Static object to get/set Session, Cookie and Querystring variables
    /// </summary>
    public partial class LocalBase
    {
        private static HttpContext _context;
        public static HttpContext Context
        {
            get
            {
                return HttpContext.Current;
                //if (_context == null)
                //    _context = HttpContext.Current;
                //return HttpContext.Current == null || HttpContext.Current.Session == null
                //    ? _context
                //    : HttpContext.Current;
            }
            set
            {
                _context = HttpContext.Current = value;
            }
        }

        public static char KeyDelimiter = '_';
        public static TimeSpan DefaultCookieLifeTime = TimeSpan.FromDays(100);
        public static TimeSpan DefaultCacheLifeTime = TimeSpan.FromMinutes(20);
        public static TimeSpan DefaultSessionLifeTime
        {
            get { return TimeSpan.FromMinutes(Context.Session.Timeout); }
            set { Context.Session.Timeout = (int)value.TotalMinutes; }
        }

        //                                      MEMCACHE
        #region MEMCACHE
        //private static Enyim.Caching.MemcachedClient _memcachedClient;

        //public static Enyim.Caching.MemcachedClient MemCachedClient
        //{
        //    get
        //    {
        //        if (_memcachedClient == null)
        //            _memcachedClient = new Enyim.Caching.MemcachedClient();
        //        return _memcachedClient;
        //    }
        //}
        #endregion
        //                                       COOKIES
        #region COOKIES
        public static bool CookiesSupported
        {
            get
            {
                if (Context == null)
                    return false;
                return Context.Request.Browser.Cookies;
            }
        }

        public static bool SetCookie<T>(string key, T value)
        {
            return SetCookie<T>(key, value, DefaultCookieLifeTime, false, false, null, null, null, null);
        }

        public static bool SetCookie<T>(string key, T value, bool compress, bool encrypt)
        {
            return SetCookie<T>(key, value, DefaultCookieLifeTime, compress, encrypt, null, null, null, null);
        }

        public static bool SetCookie<T>(string key, T value, TimeSpan lifeTime, bool compress, bool encrypt)
        {
            return SetCookie<T>(key, value, lifeTime, compress, encrypt, null, null, null, null);
        }

        public static bool SetCookie<T>(string key, T value, TimeSpan lifeTime, bool compress, bool encrypt,
            string path, string domain, bool? httpOnly, bool? secure)
        {
            string strValue = Serialization.ToJsonString<T>(value);
            return SetCookie(key, strValue, lifeTime, compress, encrypt, path, domain, httpOnly, secure);
        }

        public static bool SetCookie(string key, string value)
        {
            return SetCookie(key, value, DefaultCookieLifeTime, false, false, null, null, null, null);
        }

        public static bool SetCookie(string key, string value, bool compress, bool encrypt)
        {
            return SetCookie(key, value, DefaultCookieLifeTime, compress, encrypt, null, null, null, null);
        }

        public static bool SetCookie(string key, string value, TimeSpan lifeTime)
        {
            return SetCookie(key, value, lifeTime, false, false, null, null, null, null);
        }

        public static bool SetCookie(string key, string value, TimeSpan lifeTime, bool compress, bool encrypt)
        {
            return SetCookie(key, value, lifeTime, compress, encrypt, null, null, null, null);
        }

        public static bool SetCookie(string key, string value, TimeSpan lifeTime, bool compress, bool encrypt,
            string path, string domain, bool? httpOnly, bool? secure)
        {
            if (!CookiesSupported)
                return false;

            HttpCookie cookie;
            HttpResponse response = Context.Response;
            HttpRequest request = Context.Request;

            try
            {
                // Compression is applied before encryption since the unencrypted text usually
                // contains more repetitive patterns the compression can use.
                if (compress)
                    value = Compression.CompressToBase64(value);

                if (encrypt)
                    value = CookieEncryption.Encrypt(value);

                //value = HttpUtility.HtmlAttributeEncode(value);

                // Split our key by the key delimiter. Any delimiter character after the 1st one is ignored.
                string[] keys = key.Split(new char[] { KeyDelimiter }, 2);
                // Determine if the key contains a subkey
                bool keyContainsParentAndChild = keys.Length > 1;

                if (!keyContainsParentAndChild)
                {
                    cookie = response.Cookies[key];
                    // This also resets the expiry date
                    cookie.Value = value;
                    // The request is also updated in case it is accessed before the response completes
                    request.Cookies[key].Value = value;
                }
                else // The key provided is in the {parent}_{child} format
                {
                    cookie = response.Cookies[keys[0]];
                    // This also resets the expiry date
                    cookie[keys[1]] = value;
                    // The request is also updated in case it is accessed before the response completes
                    request.Cookies[keys[0]][keys[1]] = value;
                }

                if (value == null)
                    cookie.Expires = DateTime.Now.AddDays(-1);
                else
                    cookie.Expires = DateTime.Now.Add(lifeTime);

                if (path != null)
                    cookie.Path = path;

                if (domain != null)
                    cookie.Domain = domain;

                if (httpOnly.HasValue)
                    cookie.HttpOnly = httpOnly.Value;

                if (secure.HasValue)
                    cookie.Secure = secure.Value;

                return true;
            }
            catch (Exception ex)
            {
                return false;
            }
        }

        public static T GetCookie<T>(string key)
        {
            return GetCookie<T>(key, false, false);
        }

        public static T GetCookie<T>(string key, bool uncompress, bool decrypt)
        {
            return Serialization.FromJsonString<T>(GetCookie(key, uncompress, decrypt));
        }

        public static string GetCookie(string key) { return GetCookie(key, false, false); }

        public static string GetCookie(string key, bool uncompress, bool decrypt)
        {
            if (!CookiesSupported) return null;
            string value = null;
            HttpCookie cookie;
            string[] keys = key.Split(new char[] { KeyDelimiter }, 2);

            try
            {
                cookie = Context.Request.Cookies[keys[0]];
            }
            catch (HttpRequestValidationException ex)
            {
                //string offensiveKey = ex.Message;
                //offensiveKey = offensiveKey.Substring(offensiveKey.IndexOf(" (") + 2);
                //offensiveKey = offensiveKey.Substring(0, offensiveKey.IndexOf("="));
                //SetCookie(offensiveKey, null);

                //Context.Response.Cookies.Clear(); // Clear all cookies from response cookie collection
                //Context.Request.Cookies.Clear(); // Clear all cookies from request cookie collection
                throw new HttpRequestValidationException(
                    ex.Message + "\n This error occured because some HTML like characters were found in the cookie." +
                    //"\n Please note that as a security measure Univar clears all cookies when this happens.",
                    ex.InnerException);
            }
            catch (Exception ex)
            {
                throw new ArgumentException(ex.Message, ex.InnerException);
            }

            if (cookie != null)
            {
                if (keys.Length < 2)
                    value = cookie.Value;
                else
                    value = cookie[keys[1]];
            }

            if (decrypt)
                value = CookieEncryption.Decrypt(value, true);

            if (uncompress)
                value = Compression.UncompressFromBase64(value, true);

            return value; // HttpUtility.HtmlDecode(value);
        }
        #endregion
        //                                        SESSIONS
        #region SESSIONS
        /// <summary>
        /// Gets and sets the amount of time, in minutes, allowed between requests before
        /// the session-state provider terminates the session.
        /// </summary>
        /// <value>The time-out period, in minutes.</value>
        public static int SessionTimeOut
        {
            get { return Context.Session.Timeout; }
            set { Context.Session.Timeout = value; }
        }

        public static string GetSession(string key)
        {
            return GetSession<string>(null, key);
        }

        public static T GetSession<T>(string key)
        {
            return GetSession<T>(null, key);
        }

        public static T GetSession<T>(Page page, string key)
        {
            //if (session == null)
            //    throw new InvalidOperationException(
            //        "The session state is null. This happens when it is accessed before the PreInit event of the page.");

            object value = (page == null ? Context.Session : page.Session)[key];
            if (value != null)
                return (T)value;
            else
                return default(T);
        }

        public static void SetSession(string key, object value)
        {
            SetSession<object>(null, key, value);
        }

        public static void SetSession<T>(string key, T value)
        {
            SetSession<T>(null, key, value);
        }

        public static void SetSession<T>(Page page, string key, T value)
        {
            if (value != null)
                (page == null ? Context.Session : page.Session)[key] = value;
            else
                (page == null ? Context.Session : page.Session).Remove(key);
        }
        #endregion
        //                                      QUERY STRINGS
        #region QUERY STRINGS
        
        public static int GetQueryStringSize()
        {
            return HttpUtility.UrlDecode(Context.Request.QueryString.ToString()).Length;
        }

        /// <summary>
        /// Gets a reference to the query HTTP string collection. Note that this reference is read-only and the next
        /// constructor must be used to obtain an editable version.
        /// </summary>
        /// <returns>A reference to the the query string collection.</returns>
        public static NameValueCollection GetQueryCollection()
        {
            return GetQueryCollection(false);
        }

        /// <summary>
        /// Gets the HTTP query string collection.
        /// </summary>
        /// <param name="cloneForEditing">When true the original collection is cloned to allow editing
        /// which is not allowed otherwise.</param>
        /// <returns>A collection representing the query string.</returns>
        public static NameValueCollection GetQueryCollection(bool cloneForEditing)
        {
            if (cloneForEditing)
                return CloneQueryCollection(Context.Request.QueryString);
            else
                return Context.Request.QueryString;
        }

        /// <summary>
        /// Gets the query string.
        /// </summary>
        /// <returns>The query string data</returns>
        public static string GetQueryString()
        {
            return Context.Request.Url.Query;
        }

        public static T GetQueryString<T>(string key)
        {
            return GetQueryString<T>(key, false, false);
        }

        public static T GetQueryString<T>(string key, bool uncompress, bool decrypt)
        {
            return Serialization.FromJsonString<T>(GetQueryString(key, uncompress, decrypt));
        }

        public static string GetQueryString(string key)
        {
            return GetQueryString(key, false, false);
        }

        public static string GetQueryString(string key, bool uncompress, bool decrypt)
        {
            // Get the the value under the specified key.
            // It took me quite a long time to figure this out but for unknown reasons, the request
            // replaces every '+' character in the query with an empty character. 
            // To compensate all spaces need to be restored as their original '+' equivalent.
            string value = Context.Request.QueryString[key];

            if (value == null)
                return null;

            value = value.Replace(" ", "+"); // See above comment.

            if (decrypt)
                value = CookieEncryption.Decrypt(value, true);

            if (uncompress)
                value = Compression.UncompressFromBase64(value, true);

            return HttpUtility.UrlDecode(value);
        }

        public static void SetQueryString(NameValueCollection queryCollection)
        {
            SetQueryString(true, queryCollection);
        }

        public static void SetQueryString(string value)
        {
            RedirectTo(Context.Request.Url.AbsolutePath, value, null);
        }

        public static void SetQueryString(string key, string value)
        {
            SetQueryString<string>(true, key, value, false, false);
        }

        public static void SetQueryString(string key, string value, bool compress, bool encrypt)
        {
            SetQueryString<string>(true, key, value, compress, encrypt);
        }

        public static void SetQueryString<T>(string key, T value)
        {
            SetQueryString<T>(true, key, value, false, false);
        }

        public static void SetQueryString<T>(string key, T value, bool compress, bool encrypt)
        {
            SetQueryString<T>(true, key, value, compress, encrypt);
        }
        /// <summary>
        /// Update the query string collection. 
        /// Note that the page is automatically refreshed for the url to be updated to the new value.
        /// </summary>
        /// <param name="mergeWithBrowserQueryString">Merge the query collection with the browser query string.</param>
        /// <param name="queryCollection">The NameValueCollection containing the new key/value pairs.</param>
        /// <remarks>The page is not refreshed if the actual query string is already equal to the new value being assigned.</remarks>
        public static void SetQueryString(bool mergeWithBrowserQueryString, NameValueCollection queryCollection)
        {
            string qs = BuildQueryString(mergeWithBrowserQueryString, queryCollection);
            // Redirect only if the browser query string is different from the the new value specified.
            if (qs != Context.Request.QueryString.ToString())
                Context.Response.Redirect(Context.Request.Url.AbsolutePath + "?" + qs);
        }

        public static void SetQueryString<T>(bool mergeWithBrowserQueryString, string key, T value)
        {
            SetQueryString<T>(mergeWithBrowserQueryString, key, value, false, false);
        }

        /// <summary>
        /// Add or modify a key/value pair in the query string.
        /// </summary>
        /// <typeparam name="T">The value type.</typeparam>
        /// <param name="mergeWithBrowserQueryString">Merge the browser query string with the given key/value pair.</param>
        /// <param name="key">The reference key.</param>
        /// <param name="value">The value to store.</param>
        /// <param name="compress">Enable value compression.</param>
        /// <param name="encrypt">Enable value encrytion.</param>
        public static void SetQueryString<T>(bool mergeWithBrowserQueryString, string key, T value, bool compress, bool encrypt)
        {
            NameValueCollection nvc = BuildQueryCollectionFrom<T>(null, key, value, compress, encrypt);
            SetQueryString(mergeWithBrowserQueryString, nvc);
        }

        public static NameValueCollection BuildQueryCollection(bool mergeWithBrowserQueryString, string key, string value)
        {
            return BuildQueryCollectionFrom<string>(
                mergeWithBrowserQueryString ? Context.Request.QueryString : null, key, value, false, false);
        }

        public static NameValueCollection BuildQueryCollection<T>(bool mergeWithBrowserQueryString, string key, T value, bool compress)
        {
            return BuildQueryCollectionFrom<T>(
                mergeWithBrowserQueryString ? Context.Request.QueryString : null, key, value, false, false);
        }

        public static NameValueCollection BuildQueryCollection<T>(bool mergeWithBrowserQueryString, string key, T value, bool compress, bool encrypt)
        {
            return BuildQueryCollectionFrom<T>(
                 mergeWithBrowserQueryString ? Context.Request.QueryString : null, key, value, compress, encrypt);
        }

        public static NameValueCollection BuildQueryCollectionFrom(NameValueCollection sourceQueryCollection,
            string key, string value)
        {
            return BuildQueryCollectionFrom<string>(sourceQueryCollection, key, value, false, false);
        }

        public static NameValueCollection BuildQueryCollectionFrom(NameValueCollection sourceQueryCollection,
            string key, string value, bool compress, bool encrypt)
        {
            return BuildQueryCollectionFrom<string>(sourceQueryCollection, key, value, compress, encrypt);
        }

        public static NameValueCollection BuildQueryCollectionFrom<T>(NameValueCollection sourceQueryCollection,
            string key, T value, bool compress, bool encrypt)
        {
            NameValueCollection nvc;
            string queryValue = Serialization.ToJsonString<T>(value, Format.None, false);

            // Compression is applied before encryption since the unencrypted text usually
            // contains more repetitive patterns the compression can use.
            if (compress)
                queryValue = Compression.CompressToBase64(queryValue);

            if (encrypt)
                queryValue = CookieEncryption.Encrypt(queryValue);

            if (sourceQueryCollection == null)
            {
                nvc = new NameValueCollection();
                nvc.Add(key, queryValue);
            }
            else
            {
                nvc = CloneQueryCollection(sourceQueryCollection);
                if (value != null)
                    if (nvc[key] == null)
                        nvc.Add(key, queryValue);
                    else
                        nvc[key] = queryValue;
                else
                    nvc.Remove(key); // Keys are removed by setting their values as null.
            }
            return nvc;
        }


        /// <summary>
        /// Construct a query string from a collection.
        /// </summary>
        /// <param name="clearActualQueryString">Discard any information in the browser query string.</param>
        /// <param name="keyValuePairs">The key/value pairs used to construct the query string.</param>
        /// <returns></returns>
        public static string BuildQueryString(bool mergeWithBrowserQueryString, NameValueCollection queryCollection)
        {
            NameValueCollection nvc = new NameValueCollection();
            if (mergeWithBrowserQueryString)
                nvc = Context.Request.QueryString;

            for (int i = 0; i < queryCollection.Count; i++)
                nvc = BuildQueryCollectionFrom<string>(nvc, queryCollection.Keys[i], queryCollection[i], false, false);

            string queryText = "";
            for (int i = 0; i < nvc.Count; i++)
                queryText += string.Format("{0}{1}={2}", i == 0 ? "" : "&", nvc.Keys[i], nvc[i]);

            if (queryText.Length > 2048)
            {
                throw new Exception(
                    "The size of the query string has reached " + queryText.Length + "(Maximum allowed is 2048).");
            }

            return queryText;// HttpUtility.UrlEncode(queryText);
        }

        /// <summary>
        /// Construct a query string from an array of successive key/value pairs.
        /// </summary>
        /// <param name="clearActualQueryString">Discard any information in the browser query string.</param>
        /// <param name="keyValuePairs">The key/value pairs used to construct the query string.</param>
        /// <returns></returns>
        public static string BuildQueryString(bool mergeWithBrowserQueryString, params object[] keyValuePairs)
        {
            // Get an editable copy of the query string collection which does not allow editing.
            NameValueCollection nvc = new NameValueCollection();
            if (keyValuePairs != null)
            {
                for (int i = 0; i < keyValuePairs.Length; i = i + 2)
                {
                    if (keyValuePairs[i + 1] != null) // Exclude keys with null values
                    {
                        if (nvc[keyValuePairs[i].ToString()] == null)
                            nvc.Add(keyValuePairs[i].ToString(), keyValuePairs[i + 1].ToString());
                        else
                            nvc[keyValuePairs[i].ToString()] = keyValuePairs[i + 1].ToString();
                    }
                }
            }
            return BuildQueryString(mergeWithBrowserQueryString, nvc);
        }

        /// <summary>
        /// Creates an editable clone of the query string.
        /// This function is used as a workaround for the fact that the default query string collection is readonly.
        /// </summary>
        /// <param name="sourceQueryString">The query string collection to copy</param>
        /// <returns>An editable clone of the query string</returns>
        private static NameValueCollection CloneQueryCollection(NameValueCollection sourceQueryCollection)
        {
            if (sourceQueryCollection == null)
            {
                return null;
            }
            else
            {
                NameValueCollection nvc = new NameValueCollection();
                nvc.Add(sourceQueryCollection);
                return nvc;
            }
        }
        /// <summary>
        /// Copy all the query string to the session.
        /// </summary>
        /// <param name="clearQueryString">Clear the query string when done.</param>
        public static void CopyQueryStringToSession(bool clearQueryString)
        {
            foreach (KeyValuePair<string, string> keyValue in Context.Request.QueryString)
                SetSession(keyValue.Key, keyValue.Value);

            if (clearQueryString)
                ClearQueryString();
        }
        /// <summary>
        /// Copy all the query string to the cookie.
        /// </summary>
        /// <param name="lifetime">The life time of the cookie</param>
        /// <param name="clearQueryString">Clear the query string when done.</param>
        public static void CopyQueryStringToCookie(TimeSpan lifetime, bool clearQueryString)
        {
            foreach (KeyValuePair<string, string> keyValue in Context.Request.QueryString)
                SetCookie(keyValue.Key, keyValue.Value, lifetime);

            if (clearQueryString)
                ClearQueryString();
        }

        /// <summary>
        /// Clears the query string along with the anchor tag.
        /// </summary>
        public static void ClearQueryString()
        {
            Context.Response.Redirect(Context.Request.Url.AbsolutePath);
        }

        /// <summary
        /// Gets the anchor string from a given URL.
        /// </summary>
        /// <param name="url">The URL from which to get the anchor tag. 
        /// Note that this method is only limited to parsing an input string. The anchor tag on the browser 
        /// however is never sent as part of the HTTP request and is therefore inaccessible from the server side.
        /// </param>
        /// <returns>The anchor tag.</returns>
        public static string GetAnchor(string url)
        {
            string[] urlParts = url.Split('#');
            return urlParts.Length > 1 ? urlParts[1] : null;
        }

        /// <summary>
        /// Sets the anchor to a specific control on a page.
        /// </summary>
        /// <param name="control">The control to which the page will be anchored to.</param>
        public static void SetAnchor(System.Web.UI.Control control)
        {
            RedirectTo(Context.Request.Url.PathAndQuery, "", control.ClientID);
        }

        /// <summary>
        /// Sets the anchor value in the page URL.
        /// </summary>
        /// <param name="anchor">The client control ID to which the page will be anchored to.</param>
        public static void SetAnchor(string anchor)
        {
            RedirectTo(Context.Request.Url.PathAndQuery, "", anchor);
        }

        public static void RedirectTo(string url, string queryString) { RedirectTo(url, queryString, null); }

        public static void RedirectTo(string url, string queryString, string anchor)
        {
            Context.Response.Redirect(CreateUrl(url, queryString.ToString(), anchor));
        }

        //public static void RedirectTo(string url, string queryString, string anchor, int width, int height)
        //{
        //    Context.Response.Redirect(CreateUrl(url, queryString.ToString(), anchor), "_blank",
        //        string.Format("menubar=0,width={0},height={1}", width, height));
        //}

        public static string CreateUrl(string baseUrl, string queryString, string anchor)
        {
            if (!string.IsNullOrEmpty(queryString))
                baseUrl = baseUrl.TrimEnd('?') + "?" + queryString.TrimStart('?');
            if (!string.IsNullOrEmpty(anchor))
                baseUrl = baseUrl.TrimEnd('#') + "#" + anchor.TrimStart('#');
            return baseUrl;
        }

        public static void RefreshPage()
        {
            Context.Response.Redirect(Context.Request.Url.PathAndQuery, true);
        }
        #endregion
        //                                         CACHE
        #region CACHE
        public static void SetCache(bool shared, string key, object value)
        {
            SetCache<object>(null, shared, key, value, DefaultCookieLifeTime, null, CacheItemPriority.Normal, null);
        }

        public static void SetCache<T>(bool shared, string key, T value)
        {
            SetCache<T>(null, shared, key, value, DefaultCookieLifeTime, null, CacheItemPriority.Normal, null);
        }

        public static void SetCache<T>(bool shared, string key, T value, TimeSpan lifeTime)
        {
            SetCache<T>(null, shared, key, value, lifeTime, null, CacheItemPriority.Normal, null);
        }

        /// <summary>
        /// Saves a value to the cache. The key associated with the value is associated with the session ID such that
        /// it is not shared with other users.
        /// </summary>
        /// <typeparam name="T">The object type.</typeparam>
        /// <param name="Page">The page from which the request is made.</param>
        /// <param name="key">The key under which the value is to be saved.</param>
        /// <param name="value">The value to save.</param>
        /// <param name="lifeTime">The lifetime of the cache object.</param>
        /// <param name="cacheDependencies">The file or cache key dependencies for the cache object that determine when it must be removed.</param>
        /// <param name="cacheItemPriority">The cache priority that the cache object is given to determine when it must be removed.</param>
        /// <param name="cacheItemRemovedCallback">The event that is called whenever the cache object is removed.</param>
        public static void SetCache<T>(Page page, bool shared, string key, T value, TimeSpan lifeTime, CacheDependency cacheDependencies,
            CacheItemPriority cacheItemPriority, CacheItemRemovedCallback cacheItemRemovedCallback)
        {
            if (!shared)
                key = (page != null ? page.Session : Context.Session).SessionID + KeyDelimiter + key;
            if (value != null)
                HttpRuntime.Cache.Insert(key, value, cacheDependencies, DateTime.Now.Add(lifeTime),
                  System.Web.Caching.Cache.NoSlidingExpiration, cacheItemPriority, cacheItemRemovedCallback);
            else
                HttpRuntime.Cache.Remove(key);
        }

        public static string GetCache(bool shared, string key)
        {
            return GetCache<string>(null, shared, key);
        }

        public static T GetCache<T>(bool shared, string key)
        {
            return GetCache<T>(null, shared, key);
        }

        public static T GetCache<T>(Page page, bool shared, string key)
        {
            if (!shared)
                key = (page != null ? page.Session : Context.Session).SessionID + KeyDelimiter + key;
            object value = HttpRuntime.Cache[key];
            return value == null ? default(T) : (T)value;
        }
        #endregion
        //                                     AUTHENTICATION
        #region AUTHENTICATION
        public static bool UserIsAuthenticated()
        {
            return (Context.User != null
                && Context.User.Identity != null
                && !string.IsNullOrEmpty(Context.User.Identity.Name));
        }

        public static string UserName
        {
            get
            {
                if (Context.User.Identity.IsAuthenticated)
                    return Context.User.Identity.Name;
                // If we have never seen them return the current anonymous key for the user
                string userName = GetCookie("Userkey");
                return !string.IsNullOrEmpty(userName) ? userName : Context.Profile.UserName;
            }
            set
            {
                SetCookie("Userkey", value, DefaultCookieLifeTime);
            }
        }
        #endregion
    }
}