﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Reflection;
using System.Web.Caching;
using System.Web.UI;

namespace System.Web
{
    public partial class Local : LocalBase
    {
        /// <summary>
        /// A class used to read/write local variables from any local sources like a cookie, query string,
        /// session or cache.
        /// </summary>
        /// <typeparam key="T">The generic type T for the variable.</typeparam>
        public class Var<T>
        {
            protected string _key;
            public Sources[] _sourceTypes;
            public int? _targetTypes;
            private T _buffer;
            /// <summary>
            /// A reference to the Page control used for asynchronous session and cache operations.
            /// Note that for some delegates like the CacheItemRemovedCallback the reference must be made
            /// before the asynchronous method is called within a method like the Form_Load.
            /// </summary>
            public Page AsyncPageRef { get; set; }

            public T DefaultValue { get; set; }
            private bool _connected { get; set; }

            // Memcached properties
            //public TimeSpan MemcachedLifeTime { get; set; }

            // Query string properties
            // Note that compression is only recommended for data larger than 200 bytes in size
            // since the minimum size of data the compression yields is of this size. In other words
            // compressing a very small string of 10 bytes will still result in a 200 bytes long string.
            public bool CompressQueryString { get; set; }
            public bool EncryptQueryString { get; set; }
            // Cookie properties
            public bool CompressCookie { get; set; }
            public bool EncryptCookie { get; set; }
            public TimeSpan CookieLifeTime { get; set; }
            public string CookiePath { get; set; }
            public string CookieDomain { get; set; }
            public bool CookieIsHttpOnly { get; set; }
            public bool CookieIsSecure { get; set; }
            // Cache properties
            public TimeSpan CacheLifeTime { get; set; }
            public CacheItemRemovedCallback CacheItemRemovedCallback { get; set; }
            public CacheItemPriority CacheItemPriority { get; set; }
            public CacheDependency CacheDependencies { get; set; }
            public bool ShareCache { get; set; } // When true data is shared amongs all users.
            // Server cookie properties
            public string ServerCookiePath { get; set; }
            public TimeSpan? ServerCookieLifeTime { get; set; }
            public bool EncryptServerCookie { get; set; }
            public bool ShareServerCookie { get; set; } // When true data is shared amongs all users.

            /// <summary>
            /// This disables user identification via the membership provider.
            /// When true the id of the user is always retrieved from the client cookie.
            /// </summary>
            public bool DisableServerCookieMembershipIdentification { get; set; }

            public Var(string key, params Sources[] sourceTypes)
                : this(default(T), key, null, null, null, null, sourceTypes) { }

            public Var(string key, int? targetTypes, params Sources[] sourceTypes)
                : this(default(T), key, null, null, null, targetTypes, sourceTypes) { }

            public Var(T defaultValue, string key, params Sources[] sourceTypes)
                : this(defaultValue, key, null, null, null, null, sourceTypes) { }

            public Var(T defaultValue, string key, int? targetTypes, params Sources[] sourceTypes)
                : this(defaultValue, key, null, null, null, targetTypes, sourceTypes) { }

            /// <summary>
            /// Initializes a new instance of the <see cref="Var&lt;T&gt;"/> class.
            /// </summary>
            /// <param name="defaultValue">The default value used when the specified key does not exist in the 
            /// specified sources.</param>
            /// <param name="key">The key for the stored value.</param>
            /// <param name="cookieLifeTime">
            /// The cookie life time. Its value is set to 100 days when a null value is specified.</param>
            /// <param name="cacheLifeTime">
            /// The cache life time. Its value is set to 20 minutes when a null value is specified.</param>
            /// <param name="targetTypes">Specifies the storage types where the value is allowed to be saved. 
            /// Those are hardcoded in the Targets class. Several types can be specified
            /// by adding them up using the '+' operator as such: 
            /// Targets.Cookie + Targets.Session + Targets.Cache</param>
            /// <param name="sourceTypes">A list of storage types from which the value will be retrieved
            /// Each type is hardcoded within the Sources enum and they must be seperated by a comma.
            /// The order in which they are listed determines the order in which they are accessed until a value is found.
            /// </param>
            public Var(T defaultValue, string key, TimeSpan? cookieLifeTime, CacheDependency cacheDependency,
                TimeSpan? cacheLifeTime, int? targetTypeFlags, params Sources[] sourceTypes)
            {
                _connected = true;
                _key = key;
                // Use Sources.Session as default when no source is specified.
                SourceTypes = sourceTypes.Length > 0 ? sourceTypes : new Sources[] { Sources.Session };
                TargetTypes = targetTypeFlags;
                DefaultValue = defaultValue;

                CookieLifeTime = cookieLifeTime.HasValue ? cookieLifeTime.Value : Local.DefaultCookieLifeTime;
                CacheLifeTime = cacheLifeTime.HasValue ? cacheLifeTime.Value : Local.DefaultCacheLifeTime;
                CacheItemPriority = CacheItemPriority.Default;
                CacheDependencies = cacheDependency;
                ServerCookieLifeTime =
                    ServerCookieLifeTime.HasValue ? ServerCookieLifeTime.Value : Local.DefaultServerCookieLifeTime;
            }

            /// <summary>
            /// The sum of the target types in which the values is saved.
            /// </summary>
            public int? TargetTypes
            {
                get
                {
                    return _targetTypes;
                }
                set
                {
                    if (value.HasValue)
                        _targetTypes = value;
                    else
                        // Set the targets as the sources when none is specified.
                        _targetTypes = GetSourcesSum(SourceTypes);
                }
            }

            /// <summary>
            /// An array of sources to be used then retrieving the value.
            /// </summary>
            public Sources[] SourceTypes
            {
                get
                {
                    return _sourceTypes;
                }
                set
                {
                    _sourceTypes = value;
                    // Set the target types as the source types if it is null.
                    if (!TargetTypes.HasValue)
                        _targetTypes = GetSourcesSum(value);
                }
            }

            private int? GetSourcesSum(Sources[] sources)
            {
                int? sum = null;
                // Add all sources to the TargetTypes property.
                foreach (Sources source in sources)
                    // When the source type is set as None avoid assigning the target with the same value
                    // since this would make our Value property readonly.
                    if (source != Sources.None)
                        sum = (sum ?? 0) + (byte)source;
                return sum;

            }

            public string Key
            {
                get { return _key; }
            }

            public bool Connected
            {
                get { return _connected; }
            }

            public Type ValueType
            {
                get { return typeof(T); }
            }

            /// <summary>
            /// Gets the first available source. This is the source that is used to fetch the value.
            /// </summary>
            public Sources FirstAvalailableSource
            {
                get
                {
                    Sources source = Sources.None;
                    GetValueAsObject(_key, out source, _sourceTypes);
                    return source;
                }
            }

            /// <summary>
            /// Indicates whether the key exists in either of the specified sources and has a valid value.
            /// </summary>
            /// <value><c>true</c> if exists; otherwise, <c>false</c>.</value>
            public bool KeyExistsAndValueIsValid
            {
                get { return FirstAvalailableSource != Sources.None; }
            }

            /// <summary>
            /// Indicates whether the key exists in either of the sepcified sources.
            /// </summary>
            /// <value><c>true</c> if exists; otherwise, <c>false</c>.</value>
            public bool KeyExists
            {
                get
                {
                    Sources dummySource;
                    return new Var<object>(null).GetValueAsObject(Key, out dummySource, SourceTypes) != null;
                }
            }

            /// <summary>
            /// Gets the value from the first source containing the specified key.
            /// An array of sources can be specified using the SourceTypes parameter.
            /// The default source used is the session when none is specified.
            /// </summary>
            /// <param name="key">The key under which the value is stored.</param>
            /// <returns>An object representing the value.</returns>
            protected T GetValue(string key, params Sources[] sourceTypes)
            {
                Sources source;
                return GetValue(key, out source, sourceTypes);
            }

            /// <summary>
            /// Gets the value from the first source containing the specified key.
            /// An array of sources can be specified using the SourceTypes parameter.
            /// The default source used is the session when none is specified.
            /// </summary>
            /// <param name="key">The key under which the value is stored.</param>
            /// <param name="source">The storage type from which the value was retrieved.
            /// Sources.None is returned when the key is inexistant.</param>
            /// <returns>An object representing the value.</returns>
            protected T GetValue(string key, out Sources source, params Sources[] sourceTypes)
            {
                object value = GetValueAsObject(key, out source, sourceTypes);
                if (value != null)
                {
                    if (source == Sources.Cookie || source == Sources.QueryString || source == Sources.ServerCookie)
                        return Serialization.FromJsonString<T>(value.ToString());
                    else
                        return (T)value;
                }

                source = Sources.None;
                return default(T);
            }

            ///// <summary>
            ///// Gets the value from the first source containing the specified key as a plain object.
            ///// An array of sources can be specified using the SourceTypes parameter.
            ///// The default source used is the session when none is specified.
            ///// </summary>
            ///// <param name="key">The key under which the value is stored.</param>
            ///// <param name="source">Returns the storage type from which the value was retrieved.
            ///// Sources.None is returned when the key is inexistant.</param>
            ///// <returns>An object representing the value.</returns>
            protected object GetValueAsObject(string key, out Sources source, params Sources[] sourceTypes)
            {
                object value = null;
                // Loop through each source type until the specified key is found.
                foreach (Sources sourceType in sourceTypes)
                {
                    switch (sourceType)
                    {
                        case Sources.Session:
                            value = GetSession<object>(AsyncPageRef, key);
                            break;
                        case Sources.Cookie:
                            value = GetCookie(key, CompressCookie, EncryptCookie);
                            break;
                        case Sources.Cache:
                            value = GetCache<object>(AsyncPageRef, ShareCache, key);
                            break;
                        case Sources.ServerCookie:
                            value = GetServerCookie(ServerCookiePath, ShareServerCookie, key,
                                DisableServerCookieMembershipIdentification, EncryptServerCookie);
                            break;
                        //case Sources.Memcached:
                        //    value = MemCachedClient.Get(key);
                        //    break;
                        case Sources.QueryString:
                            value = GetQueryString(key, CompressQueryString, EncryptQueryString);
                            break;
                    }

                    if (value != null)
                    {
                        source = sourceType;
                        return value;
                    }
                }

                source = Sources.None;
                return null;
            }

            /// <summary>
            /// Set the specified value to the targets in the TargetTypes flags.
            /// </summary>
            /// <typeparam name="T">The type of the value.</typeparam>
            /// <param name="key">The key under which the value is stored.</param>
            /// <param name="value">The value to be stored.</param>
            public void SetValue(string key, T value)
            {
                int flags = TargetTypes ?? 0;

                if ((flags & Targets.Session) != 0)
                    SetSession(AsyncPageRef, key, value);

                if ((flags & Targets.Cookie) != 0)
                    SetCookie<T>(key, value, CookieLifeTime, CompressCookie, EncryptCookie,
                        CookiePath, CookieDomain, CookieIsHttpOnly, CookieIsSecure);

                if ((flags & Targets.Cache) != 0)
                    SetCache(AsyncPageRef, ShareCache, key, value, CacheLifeTime, CacheDependencies, CacheItemPriority, CacheItemRemovedCallback);

                if ((flags & Targets.ServerCookie) != 0)
                    SetServerCookie<T>(ServerCookiePath, ShareServerCookie, key, value, DisableServerCookieMembershipIdentification,
                        ServerCookieLifeTime, EncryptServerCookie);

                //if ((flags & Targets.Memcached) != 0)
                //    MemCachedClient.Store(Enyim.Caching.Memcached.StoreMode.Set, key, value);

                //if ((flags & Targets.Anchor) != 0)
                //    SetAnchor( BuildQueryString(true, key, value).ToString());

                // The query string is always set last since it uses the Redirect method to do so.
                // This disrupt the flow and therefore cause any code coming after to be skipped.
                if ((flags & Targets.QueryString) != 0)
                    SetQueryString<T>(key, value, CompressQueryString, EncryptQueryString);

            }


            public T Value
            {
                get
                {
                    if (!_connected)
                        return _buffer;
                    else
                    {
                        Sources source;
                        T value = GetValue(_key, out source, _sourceTypes);

                        if (source != Sources.None)
                            return value;
                        else
                            return DefaultValue;
                    }
                }
                set
                {
                    if (!_connected)
                        _buffer = value;
                    else
                        SetValue(_key, value);
                }
            }

            /// <summary>
            /// When disconnected all read/write operations are carried out directly in memory until
            /// the connection is restored. This helps minimize the overhead of accessing the specified
            /// storage for each operation.
            /// </summary>
            public void Disconnect()
            {
                _buffer = GetValue(_key, _sourceTypes);
                _connected = false;
            }

            public void SaveChanges()
            {
                SaveChanges(false);
            }

            public void RestoreConnectionAndSaveChanges()
            {
                SaveChanges(true);
            }
            /// <summary>
            /// Saves the changes applied to the buffer from the last disconnection.
            /// </summary>
            /// <param name="restoreConnection">When true the connection is restored.
            /// Note that the connection is always restored even if an error occurs
            /// during the operation.</param>
            public void SaveChanges(bool restoreConnection)
            {
                if (restoreConnection)
                    _connected = true;
                SetValue(_key, _buffer);
            }

            /// <summary>
            /// Concatenates an array of enums into a string.
            /// </summary>
            /// <param name="enums">The array of enums to concatenate</param>
            /// <returns></returns>
            private string ConcatenateEnums(params Enum[] enums)
            {
                string value = string.Empty;
                for (int i = 0; i < enums.Length; i++)
                    value += enums[i].ToString() + (i < enums.Length - 1 ? KeyDelimiter.ToString() : "");
                return value;
            }

            /// <summary>
            /// Represents the value under the concatenated parent and child keys.
            /// </summary>
            /// <param name="childKey">The child keys to be appended to the parent key using 
            /// KeyDelimiter as seperator.</param>
            /// <returns></returns>
            public T this[params Enum[] childKeys]
            {
                get { return this[ConcatenateEnums(childKeys)]; }
                set { this[ConcatenateEnums(childKeys)] = value; }
            }

            /// <summary>
            /// Represents the value under the concatenated parent and child keys.
            /// </summary>
            /// <param name="childKey">The child key to be appended to the parent key using 
            /// KeyDelimiter as seperator.</param>
            /// <returns></returns>
            public T this[double childKey]
            {
                get { return this[childKey.ToString()]; }
                set { this[childKey.ToString()] = value; }
            }

            public T this[string childKey]
            {
                get
                {
                    return GetValue(_key + KeyDelimiter + childKey, _sourceTypes);
                }
                set
                {
                    SetValue(_key + KeyDelimiter + childKey, value);
                }
            }

            public override string ToString()
            {
                return Serialization.ToJsonString<T>(Value);
            }

            public string ToString(Format encoder, bool serializePrimitives)
            {
                return Serialization.ToJsonString<T>(Value, encoder, serializePrimitives);
            }

            public void LoadValueFromString(string serializedText, bool deserializePrimitives)
            {
                this.Value = Serialization.FromJsonString<T>(serializedText, Format.None, deserializePrimitives);
            }

            /// <summary>
            /// Copies the value from the first available source in the sources specified to all the target types.
            /// </summary>
            public void CopySourceToAllTargets()
            {
                this.Value = GetValue(_key, _sourceTypes);
            }

            public void CopySourceTo(int targetTypes)
            {
                Var<T> dummyVar = this.Clone();
                dummyVar.TargetTypes = targetTypes;
                dummyVar.CopySourceToAllTargets();
            }

            public void MoveAllSourcesTo(int targetTypes)
            {
                T value = this.Value;
                this.Value = DefaultValue;
                Var<T> dummyVar = this.Clone();
                dummyVar.TargetTypes = targetTypes;
                dummyVar.Value = value;
            }

            public void ClearSources()
            {
                foreach (Sources source in SourceTypes)
                    new Var<object>(_key, source).Value = null;
            }

            /// <summary>
            /// Removes the key and its value in the specified target types.
            /// </summary>
            public void ClearTargets()
            {
                new Var<object>(_key, TargetTypes, Sources.None).Value = null;
            }

            public bool TargetIs(int targetType)
            {
                return (TargetTypes & targetType) != 0;
            }

            public bool SourceIs(Sources sourceType)
            {
                return SourceTypes.Contains(sourceType);
            }

            public Var<T> Clone()
            {
                return (Var<T>)this.MemberwiseClone();
            }

            /// <summary>
            /// Gets the compression level achieved when compression is enabled.
            /// </summary>
            /// <returns>The compression level as a percentage</returns>
            public int GetCompressionLevel()
            {
                string jsonValue = Serialization.ToJsonString<T>(Value);
                string compressedJsonValue = Compression.CompressToBase64(jsonValue);
                return 100 * (compressedJsonValue.Length / jsonValue.Length);
            }
        }
    }
}