﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using DevOrigin.ToDoList.Data.SqlRepository;

namespace DevOrigin.ToDoList.Data
{
  public class SqlTaskItemRepository : ITaskItemRepository
  {
    LinqTaskItemDataContext _db;

    /// <summary>
    /// One-arg CTOR for initialising the data context to be used for
    /// various methods.
    /// </summary>
    /// <param name="dataContext">TaskItem datacontext.</param>
    public SqlTaskItemRepository(LinqTaskItemDataContext dataContext)
    {
      //override the current context
      //with the one passed in
      _db = dataContext;
    }

    #region IToDoItemRepository Members

    /// <summary>
    /// Linq to SQL implementation for ToDoItem.
    /// </summary>
    /// <returns>TaskItem as Querable.</returns>
    public IQueryable<TaskItem> GetItems()
    {
      var items = from todoItem in _db.TaskItems
                  select new TaskItem
                  {
                    ItemId = todoItem.ItemID,
                    Title = todoItem.Title,
                    Priority = todoItem.Priority,
                    StartDate = todoItem.StartDate,
                    DueDate = todoItem.DueDate,
                    PercentCompleted = todoItem.PercentCompleted,
                    TrackProgress = todoItem.TrackProgress,
                    RemindOnDelay = todoItem.RemindOnDelay,
                    IsAccomplished = todoItem.IsAccomplished
                  };

      return items;
    }

    /// <summary>
    /// Sets the status of the task-item as accomplished.
    /// </summary>
    /// <param name="itemId">ID of the task-item to be accomplished</param>
    public void AccomplishTaskItem(int itemId)
    {
      using (LinqTaskItemDataContext db = new LinqTaskItemDataContext())
      {
        SqlRepository.TaskItem dbItem = db.TaskItems
          .Where(x => x.ItemID == itemId).SingleOrDefault();

        if (dbItem == null)
        {
          throw new ArgumentNullException("Unable to iterate the item in the database.");
        }

        //Update item in task-item
        dbItem.IsAccomplished = true;
        dbItem.PercentCompleted = 100;
        db.SubmitChanges();

        //Insert item in accomplished task-item
        SqlRepository.TrashedTaskItem trashedItem = db.TrashedTaskItems
          .Where(x => x.ItemID == itemId).SingleOrDefault();
        bool isNew = false;

        if (trashedItem == null)
        {
          trashedItem = new DevOrigin.ToDoList.Data.SqlRepository.TrashedTaskItem();
          isNew = true;
        }

        trashedItem.ItemID = dbItem.ItemID;
        trashedItem.Title = dbItem.Title;
        trashedItem.Priority = dbItem.Priority;
        trashedItem.StartDate = dbItem.StartDate;
        trashedItem.DueDate = dbItem.DueDate;
        trashedItem.PercentCompleted = dbItem.PercentCompleted;
        trashedItem.TrackProgress = dbItem.TrackProgress;
        trashedItem.RemindOnDelay = dbItem.RemindOnDelay;
        trashedItem.IsAccompolished = dbItem.IsAccomplished;

        if (isNew)
          db.TrashedTaskItems.InsertOnSubmit(trashedItem);

        db.SubmitChanges();
      }
    }

    /// <summary>
    /// Insert the param to database.
    /// </summary>
    /// <param name="item">TaskItem object to be inserted.</param>
    public void InsertTaskItem(TaskItem item)
    {
      if (!item.IsValid)
        throw new ArgumentException();

      using (LinqTaskItemDataContext db = new LinqTaskItemDataContext())
      {
        SqlRepository.TaskItem dbItem = db.TaskItems
          .Where(x => x.ItemID == item.ItemId)
          .SingleOrDefault();
        bool isNew = false;

        if (dbItem == null)
        {
          dbItem = new DevOrigin.ToDoList.Data.SqlRepository.TaskItem();
          isNew = true;
        }

        int itemId = (from taskItem in db.TaskItems
                      select taskItem.ItemID).Max();

        //add the item
        dbItem.ItemID = ++itemId;
        dbItem.Title = item.Title;
        dbItem.Priority = item.Priority;
        dbItem.StartDate = item.StartDate;
        dbItem.DueDate = item.DueDate;
        dbItem.PercentCompleted = item.PercentCompleted;
        dbItem.TrackProgress = item.TrackProgress;
        dbItem.RemindOnDelay = item.RemindOnDelay;

        if (isNew)
          db.TaskItems.InsertOnSubmit(dbItem);

        db.SubmitChanges();
      }
    }

    /// <summary>
    /// Update the param to the database.
    /// </summary>
    /// <param name="item">TaskItem object to be updated.</param>
    public void UpdateTaskItem(TaskItem item)
    {
      if (!item.IsValid)
        throw new ArgumentException();

      using (LinqTaskItemDataContext db = new LinqTaskItemDataContext())
      {
        SqlRepository.TaskItem dbItem = db.TaskItems
            .Where(x => x.ItemID == item.ItemId)
            .SingleOrDefault();

        if (dbItem == null)
        {
          throw new ArgumentNullException("Unable to iterate the item in the database.");
        }

        //update the item
        //dbItem.ItemID = item.ItemId; //operation not available
        dbItem.Title = item.Title;
        dbItem.Priority = item.Priority;
        dbItem.StartDate = item.StartDate;
        dbItem.DueDate = item.DueDate;
        dbItem.PercentCompleted = item.PercentCompleted;
        dbItem.TrackProgress = item.TrackProgress;
        dbItem.RemindOnDelay = item.RemindOnDelay;

        db.SubmitChanges();
      }
    }

    /// <summary>
    /// Delete a task from database.
    /// </summary>
    /// <param name="itemId">ID of the task to be deleted.</param>
    public void Delete(int itemId)
    {
      _db.TaskItems.DeleteOnSubmit(_db.TaskItems
                          .SingleOrDefault(item => item.ItemID == itemId));
      _db.SubmitChanges();
    }

    #endregion
  }
}
