﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Web;
using System.Web.UI;
using System.Text.RegularExpressions;

namespace DebugModule
{
    public static class DebugControlInjection
    {
        private const string DebugPropertyBagKeyFormat = "Debug_PropertyBag{0}";
        private static readonly Regex TagIdRegex = new Regex(@"[^<]*?<.*?\bid\b[ ]?=[ ]?[\""']?([^ \""']*)", RegexOptions.Singleline | RegexOptions.IgnoreCase);

        public static void AddControlDebug(Page page)
        {
            page.Controls.Add(new HtmlComment() { Text = "Start Debug Injection - Controls" });
            AddDebugControl(page, page.Controls);
            page.Controls.Add(new HtmlComment() { Text = "End Debug Injection - Controls" });
        }

        public static string GetClientID(Control control)
        {
            // literal controls don't get an auto-generated ClientID
            return control is LiteralControl ? control.ID : control.ClientID;
        }

        public static DebugControlItem GetDebugControlItem(UserControl control)
        {
            return control.Page.Items[string.Format(DebugPropertyBagKeyFormat, control.UniqueID)] as DebugControlItem;
        }

        #region Private Methods

        private static void EnsureClientID(UserControl uc)
        {
            LiteralControl lit = uc.Controls[0] as LiteralControl;

            if (lit != null)
            {
                Match m = TagIdRegex.Match(lit.Text);

                string id = m.Groups.Count != 0 ? m.Groups[1].Value : null;

                if (string.IsNullOrEmpty(id))
                {
                    // no id found, generate one and add it to the html
                    id = GenerateDebugId();
                    int pos = lit.Text.IndexOf('>');
                    lit.Text = string.Format("{0} id=\"{2}\" {1}", lit.Text.Substring(0, pos), lit.Text.Substring(pos), id);
                    lit.ID = id;
                }
                else
                {
                    uc.Controls[0].ID = id;
                }

            }
            else if (string.IsNullOrEmpty(uc.Controls[0].ID))
            {
                uc.Controls[0].ID = GenerateDebugId();
            }
        }

        private static void AddDebugControl(Page page, ControlCollection controls)
        {
            int count = controls.Count;
            for (int i = 0; i < count; i++)
            {
                Control c = controls[i];
                UserControl uc = c as UserControl;

                if (uc != null && uc.Visible && uc.Controls.Count != 0)
                {
                    page.Controls.Add(new DebugPaneControl() { ControlToDebug = uc });

                    EnsureClientID(uc);
                    CreateDebugControlItem(uc);
                }

                AddDebugControl(page, c.Controls);
            }
        }

        private static string GenerateDebugId()
        {
            return "debug_" + Guid.NewGuid().ToString("N");
        }

        private static void CreateDebugControlItem(UserControl control)
        {
            DebugControlItem debugItem = new DebugControlItem();
            debugItem.Properties.Add("Path", control.AppRelativeVirtualPath);
            debugItem.Properties.Add("ID", control.ID);
            debugItem.Properties.Add("ViewState Enabled", control.EnableViewState.ToString());

            control.Page.Items[string.Format(DebugPropertyBagKeyFormat, control.UniqueID)] = debugItem;
        }

        #endregion
    }
}
