/**
 * MyTableGrid, version 1.0.5
 *
 * Dual licensed under the MIT and GPL licenses.
 *
 * Copyright 2009 Pablo Aravena, all rights reserved.
 * http://pabloaravena.info/mytablegrid
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */
var MyTableGrid = Class.create();

MyTableGrid.prototype = {
    version: '1.0.5',

    imageRefs : {
        colMoveTop: 'images/mtg/col-move-top.gif',
        colMoveBottom: 'images/mtg/col-move-bottom.gif',
        sortAscendingIcon: 'images/mtg/sort_ascending.png',
        sortDescendingIcon: 'images/mtg/sort_descending.png',
        prevIcon: 'images/mtg/prev.gif',
        nextIcon: 'images/mtg/next.gif',
        firstIcon: 'images/mtg/first.gif',
        lastIcon: 'images/mtg/last.gif',
        prevDisabledIcon: 'images/mtg/prev-disabled.gif',
        nextDisabledIcon: 'images/mtg/next-disabled.gif',
        firstDisabledIcon: 'images/mtg/first-disabled.gif',
        lastDisabledIcon: 'images/mtg/last-disabled.gif',
        ySepIcon: 'images/mtg/y-sep.png',
        mtgLoaderIcon: 'images/mtg/mtg-loader.gif',
        mtgSortArrowAscIcon: 'images/mtg/sort-arrow-asc.png',
        mtgSortArrowDescIcon: 'images/mtg/sort-arrow-desc.png',
        mtgMagnifierIcon: 'images/mtg/magnifier.png'
    },

    messages : {
        totalDisplayMsg: '<strong>{total}</strong> records found',
        rowsDisplayMsg: ', displaying <strong>{from}</strong> to <strong>{to}</strong>',
        pagePromptMsg: '<td><strong>Page:</strong></td><td>{input}</td><td>of <strong>{pages}</strong></td>',
        pagerNoDataFound: '<strong>No records found</strong>'
    },

    /**
     * MyTableGrid constructor
     */
    initialize : function(tableModel) {
        this.mtgId = $$('.myTableGrid').length + 1;
        this.tableModel = tableModel;
        this.columnModel = tableModel.columnModel || [];
        this.rows = tableModel.rows || [];
        this.options = tableModel.options || {};
        this.name = tableModel.name || '';
        this.fontSize = 11;
        this.headerHeight = 24;
        this.cellHeight = 24;
        this.pagerHeight = 24;
        this.scrollBarWidth = 18;
        this.topPos = 0;
        this.leftPos = 0;
        this.selectedHCIndex = 0;
        this.pager = this.options.pager || null;
        this.pager.pageParameter = this.options.pager.pageParameter || 'page';
        this.url = tableModel.url || null;
        this.request = tableModel.request || {};
        this.sortColumnParameter = this.options.sortColumnParameter || 'sortColumn';
        this.ascDescFlagParameter = this.options.ascDescFlagParameter || 'ascDescFlg';
        this.sortedColumnIndex = 0;
        this.onCellFocus = this.options.onCellFocus || null;
        this.onCellBlur = this.options.onCellBlur || null;
        this.modifiedRows = []; //will contain the modified row numbers
        this.afterRender = this.options.afterRender || null; //after rendering handler
        this.rowStyle = this.options.rowStyle || null; //row style handler
        this.rowClass = this.options.rowClass || null; //row class handler        
        this.renderedRows = 0; //Use for lazy rendering
        this.renderedRowsAllowed = 0; //Use for lazy rendering depends on bodyDiv height

        for (var i = 0; i < this.columnModel.length; i++) {
            if (!this.columnModel[i].hasOwnProperty('editable')) this.columnModel[i].editable = false;
            if (!this.columnModel[i].hasOwnProperty('visible')) this.columnModel[i].visible = true;
            if (!this.columnModel[i].hasOwnProperty('type')) this.columnModel[i].type = 'string';
            this.columnModel[i].positionIndex = i;
        }

        this.targetColumnId = null;
        this.editedCellId = null;

        this.gap = 2; //diff between widh and offsetWidth
        if (Prototype.Browser.WebKit) this.gap = 0;
    },

    /**
     * Renders the table grid control into a given target
     */
    render : function(target) {
        this.target = target;
        var fragment = document.createDocumentFragment();
        $(target).innerHTML = this.createTableLayout();
        var id = this.mtgId;
        this.tableDiv = $('myTableGrid' + id);
        this.headerRowDiv = $('headerRowDiv' + id);
        this.bodyDiv = $('bodyDiv' + id);
        this.innerBodyDiv = $('innerBodyDiv' + id);
        this.pagerDiv = $('pagerDiv' + id);
        this.resizeMarkerLeft = $('resizeMarkerLeft' + id);
        this.resizeMarkerRight = $('resizeMarkerRight' + id);
        this.dragColumn = $('dragColumn' + id);
        this.colMoveTopDiv = $('mtgColMoveTop' + id);
        this.colMoveBottomDiv = $('mtgColMoveBottom' + id);
        this.scrollLeft = 0;
        this.scrollTop = 0;
        this.targetColumnId = null;

        var self = this;

        Event.observe(this.bodyDiv, 'dom:dataLoaded', function(){
            $('mtgLoader'+id).show();
            if (self.rows.length > 0) {
                self.bodyTable = $('mtgBT' + id);
                self.applyCellCallbacks();
                self.applyHeaderButtons();
                self.makeAllColumnResizable();
                self.makeAllColumnDraggable();
                self.applySettingMenuBehavior();
                self.keys = new KeyTable(self);
                self.addKeyBehavior();
            }
            if (self.pager) {
                self.addPagerBehavior();
            }
            if (self.afterRender) {
                self.afterRender();
            }
            $('mtgLoader'+id).hide();
        });

        setTimeout(function() {
            self.renderedRowsAllowed = Math.floor(self.bodyDiv.clientHeight / self.cellHeight); 
            if (self.tableModel.hasOwnProperty('rows')) {
                if (self.rows.length > 0) {
                    self.innerBodyDiv.innerHTML = self.createTableBody(self.rows);
                    self.pagerDiv.innerHTML = self.updatePagerInfo();
                    self.bodyDiv.fire('dom:dataLoaded');
                } else {
                    self.pagerDiv.innerHTML = self.updatePagerInfo();
                }
            } else {
                self.retrieveDataFromUrl(1, true);
            }
        }, 0);

        if (this.options.onSave) {
            Event.observe($('mtgSaveBtn'+id), 'click', function() {
               self.options.onSave();
            });
        }
        // Adding scrolling handler
        Event.observe($('bodyDiv' + id), 'scroll', function() {
            self.syncScroll();
        });
        // Adding resize handler
        Event.observe(window, 'resize', function() {
            self.resize();
        });
    },

    /**
     * Creates the table layout
     */
    createTableLayout : function(parentNode) {
        var target = $(this.target);
        var width = this.options.width || (target.getWidth() - this.fullPadding(target,'left') - this.fullPadding(target,'right')) + 'px';
        var height = this.options.height || (target.getHeight() - this.fullPadding(target,'top') - this.fullPadding(target,'bottom')) + 'px';
        var id = this.mtgId;
        var cm = this.columnModel;
        var gap = this.gap;
        var ir = this.imageRefs;
        var cellHeight = this.cellHeight;

        this.tableWidth = parseInt(width) - 2;
        this.tableHeight = parseInt(height) - 2;

        //Calculates header width
        var hw = 0;
        for (var i = 0; i < cm.length; i++) {
            hw += parseInt(cm[i].width) + gap;
        }
        this.headerWidth = hw;
        var idx = 0;
        var html = [];
        html[idx++] = '<div id="myTableGrid'+id+'" class="myTableGrid" style="position:relative;width:'+this.tableWidth+'"px;height:'+this.tableHeight+'px;z-index:0">';

        if (this.options.title || this.options.onSave) { // Adding header title
            html[idx++] = '<div id="mtgHeaderTitle'+id+'" class="mtgHeaderRow mtgHeaderTitle" style="position:absolute;top:'+this.topPos+'px;left:'+this.leftPos+'px;width:'+(this.tableWidth - 6)+'px;height:'+(this.headerHeight - 6)+'px;padding:3px;z-index:10">';
            if (this.options.onSave) {
                html[idx++] = '<input id="mtgSaveBtn'+id+'" class="buttonpurple" type="button" name="btnSave" value="Save">'; //TODO Change CSS Class Name
            } else {
                html[idx++] = this.options.title;
            }
            html[idx++] = '</div>';
            this.topPos += this.headerHeight + 1;
        }

        // Adding Header Row
        html[idx++] = '<div id="headerRowDiv'+id+'" class="mtgHeaderRow" style="position:absolute;top:'+this.topPos+'px;left:'+this.leftPos+'px;width:'+this.tableWidth+'px;height:'+this.headerHeight+'px;padding:0;overflow:hidden;z-index:0">';
        //header row box useful for drag and drop
        html[idx++] = '<div id="mtgHRB'+id+'" style="position:relative;padding:0;margin:0;width:'+this.headerWidth+'px">';
        // Adding Header Row Cells
        html[idx++] = this.createHeaderRow();
        html[idx++] = '</div>'; // closes mtgHRB
        html[idx++] = '</div>'; // closes headerRowDiv
        this.topPos += this.headerHeight + 1;

        // Adding Body Area
        this.bodyHeight = this.tableHeight - this.headerHeight - 3;
        if (this.options.title) this.bodyHeight = this.bodyHeight - this.headerHeight - 1;
        if (this.options.pager) this.bodyHeight = this.bodyHeight - this.pagerHeight;
        html[idx++] = '<div id="bodyDiv'+id+'" class="mtgBody" style="position:absolute;top:'+this.topPos+'px;left:'+this.leftPos+'px;width:'+this.tableWidth+'px;height:'+this.bodyHeight+'px;overflow:auto;z-index:10">';
        html[idx++] = '<div id="innerBodyDiv'+id+'" class="mtgInnerBody" style="position:absolute;top:0px;left:'+this.leftPos+'px;width:'+(this.tableWidth - this.scrollBarWidth)+'px;overflow:none;z-index:0">';
        html[idx++] = '</div>'; // closes innerBodyDiv
        html[idx++] = '</div>'; // closes bodyDiv

        // Adding Pager Panel
        if (this.options.pager) {
            this.topPos += this.bodyHeight + 2;
            html[idx++] = '<div id="pagerDiv'+id+'" class="mtgPager" style="position:absolute;top:'+this.topPos+'px;left:0;bottom:0;width:'+(this.tableWidth - 2)+'px;height:'+(this.pagerHeight - 4)+'px">';
            html[idx++] = this.updatePagerInfo(true);
            html[idx++] = '</div>'; // closes Pager Div
        }

        // Adding Table Setting Button Control
        html[idx++] = '<div id="mtgSB'+id+'" class="mtgSettingButton">';
        html[idx++] = '</div>';
        // Adding Table Setting Menu
        html[idx++] = this.createSettingMenu();

        // Adding Header Button Control
        html[idx++] = '<div id="mtgHB'+id+'" class="mtgHeaderButton" style="width:14px;height:'+this.headerHeight+'px">';
        html[idx++] = '</div>';
        // Adding Header Button Menu
        html[idx++] = '<div id="mtgHBM'+id+'" class="mtgMenu">';
        html[idx++] = '<ul>';
        html[idx++] = '<li>';
        html[idx++] = '<a id="mtgSortAsc'+id+'" class="mtgMenuItem" href="javascript:void(0)">';
        html[idx++] = '<table cellspacing="0" cellpadding="0" width="100%" border="0">'
        html[idx++] = '<tr><td width="25">&nbsp;<img src="'+ir.sortAscendingIcon+'" width="16" height="16" border="0" class="mtgMenuItemIcon mtgSortAscendingIcon"></td>';
        html[idx++] = '<td>Sort Ascending</td></tr></table>';
        html[idx++] = '</a>';
        html[idx++] = '</li>';
        html[idx++] = '<li>';
        html[idx++] = '<a id="mtgSortDesc'+id+'" class="mtgMenuItem" href="javascript:void(0)">';
        html[idx++] = '<table cellspacing="0" cellpadding="0" width="100%" border="0">'
        html[idx++] = '<tr><td width="25">&nbsp;<span><img src="'+ir.sortDescendingIcon+'" width="16" height="16" border="0" class="mtgMenuItemIcon mtgSortDescendingIcon"></span></td>';
        html[idx++] = '<td>Sort Descending</td></tr></table>';
        html[idx++] = '</a>';
        html[idx++] = '</li>';
        html[idx++] = '<li class="mtgSelectAll">';
        html[idx++] = '<a class="mtgMenuItem" href="javascript:void(0)">';
        html[idx++] = '<table cellspacing="0" cellpadding="0" width="100%" border="0">'
        html[idx++] = '<tr><td width="25"><span class="mtgMenuItemChk"><input type="checkbox" id="mtgSelectAll'+id+'"></span></td>';
        html[idx++] = '<td>Select All</td></tr></table>';
        html[idx++] = '</a>';
        html[idx++] = '</li>';
        html[idx++] = '</ul>';
        html[idx++] = '</div>';

        // Adding Resize Markers
        html[idx++] = '<div id="resizeMarkerLeft'+id+'" class="mtgResizeMarker">';
        html[idx++] = '</div>';
        html[idx++] = '<div id="resizeMarkerRight'+id+'" class="mtgResizeMarker">';
        html[idx++] = '</div>';

        html[idx++] = '</div>' // closes Table Div;

        // Adding Dragging controls
        html[idx++] = '<div id="mtgColMoveTop'+id+'" style="position:absolute;width:9px;height:9px;top:0;left:0;z-index:50;visibility:hidden">';
        html[idx++] = '<img src="'+ir.colMoveTop+'" width="9" height="9">';
        html[idx++] = '</div>';

        html[idx++] = '<div id="mtgColMoveBottom'+id+'" style="position:absolute;width:9px;height:9px;top:0;left:0;z-index:50;visibility:hidden">';
        html[idx++] = '<img src="'+ir.colMoveBottom+'" width="9" height="9">';
        html[idx++] = '</div>';

        html[idx++] = '<div id="dragColumn'+id+'" class="dragColumn" style="width:100px;height:18px;">';
        html[idx++] = '<span class="columnTitle">&nbsp;</span>';
        html[idx++] = '<div class="drop-no">&nbsp;</div>';
        html[idx++] = '</div>';
        return html.join('');
    },

    /**
     * Creates the Header Row
     */
    createHeaderRow : function() {
        // Templates definition
        var thTmpl = '<th id="mtgHC{id}_{x}" width="{width}" style="width:{width}px;padding:0;margin:0" class="mtgHeaderCell mtgHC{id}">';
        var thTmplLast = '<th id="mtgHC{id}_{x}" width="{width}" style="width:{width}px;padding:0;margin:0;border-right:none" class="mtgHeaderCell mtgHC{id}">';
        var ihcTmpl = '<div id="mtgIHC{id}_{x}" class="mtgInnerHeaderCell mtgIHC{id}" style="position:relative;width:{width}px;height:{height}px;padding:3px;z-index:20">';
        var ihcTmplLast = '<div id="mtgIHC{id}_{x}" class="mtgInnerHeaderCell" style="position:relative;width:{width}px;height:{height}px;padding:3px;z-index:20">';
        var hsTmpl = '<div id="mtgHS{id}_{x}" class="mtgHS mtgHS{id}" style="position:absolute;top:0;right:-4px;width:1px;height:{height}px">';
        var siTmpl = '<img id="mtgSortIcon{id}_{x}" width="8" height="4" style="display:none">';

        var cm = this.columnModel;
        var id = this.mtgId;
        var gap = (this.gap == 0)? 2 : 0;
        var cellHeight = this.headerHeight - 6;

        var idx = 0;
        var html = [];
        html[idx++] = '<table id="mtgHRT'+id+'" width="'+this.headerWidth+'" cellpadding="0" cellspacing="0" border="0" class="mtgHeaderRowTable">';
        html[idx++] = '<thead>';
        html[idx++] = '<tr>';
        // Creating Header Cell Elements
        for (var i = 0; i < cm.length; i++) {
            var cell = cm[i];
            var temp = thTmpl.replace(/\{id\}/g, id);
            temp = temp.replace(/\{x\}/g, i);
            temp = temp.replace(/\{width\}/g, parseInt(cell.width));
            html[idx++] = temp;
            temp = ihcTmpl.replace(/\{id\}/g, id);
            temp = temp.replace(/\{x\}/g, i);
            temp = temp.replace(/\{width\}/g, (parseInt(cell.width) - 8 - gap));
            temp = temp.replace(/\{height\}/g, cellHeight);
            html[idx++] = temp;
            html[idx++] = cell.title;
            temp = siTmpl.replace(/\{id\}/g, id);
            temp = temp.replace(/\{x\}/g, i);
            html[idx++] = temp;
            temp = hsTmpl.replace(/\{id\}/g, id);
            temp = temp.replace(/\{x\}/g, i);
            temp = temp.replace(/\{height\}/g, this.headerHeight);
            html[idx++] = temp;
            html[idx++] = '</div>';
            html[idx++] = '</div>';
            html[idx++] = '</th>';
        }
        // Last Header Element
        var temp = thTmplLast.replace(/\{id\}/g, id);
        temp = temp.replace(/\{x\}/g, i);
        temp = temp.replace(/\{width\}/g, '20');
        html[idx++] = temp;
        temp = ihcTmplLast.replace(/\{id\}/g, id);
        temp = temp.replace(/\{x\}/g, i);
        temp = temp.replace(/\{width\}/g, '20');
        temp = temp.replace(/\{height\}/g, cellHeight);
        html[idx++] = temp;
        html[idx++] = '&nbsp';
        html[idx++] = '</div>';
        html[idx++] = '</th>';
        html[idx++] = '</tr>';
        html[idx++] = '</thead>';
        html[idx++] = '</table>';
        return html.join('');
    },

    /**
     * Creates the Table Body
     */
    createTableBody : function(rows) {
        var id = this.mtgId;
        var tdTmpl = '<td id="mtgC{id}_{x},{y}" height="{height}" width="{width}" style="width:{width}px;height:{height}px;padding:0;margin:0" class="mtgCell mtgC{id} mtgC{id}_{x} mtgR{id}_{y}">';
        var icTmpl = '<div id="mtgIC{id}_{x},{y}" style="width:{width}px;height:{height}px;padding:3px;text-align:{align}" class="mtgInnerCell mtgIC{id} mtgIC{id}_{x} mtgIR{id}_{y}">';
        var checkboxTmpl = '<input id="mtgInput{id}_{x},{y}" name="mtgInput{id}_{x},{y}" type="checkbox" value="{value}" class="mtgInput{id}_{x} mtgInputCheckbox">';
        var radioTmpl = '<input id="mtgInput{id}_{x},{y}" name="mtgInput{id}_{x}" type="radio" value="{value}" class="mtgInput{id}_{x} mtgInputRadio">';
        if (Prototype.Browser.Opera || Prototype.Browser.WebKit) {
            checkboxTmpl = '<input id="mtgInput{id}_{x},{y}" name="mtgInput{id}_{x},{y}" type="checkbox" value="{value}" class="mtgInput{id}_{x}">';
            radioTmpl = '<input id="mtgInput{id}_{x},{y}" name="mtgInput{id}_{x}" type="radio" value="{value}" class="mtgInput{id}_{x}">';
        }
        var rs = this.rowStyle || function(){return '';}; // row style handler
        var rc = this.rowClass || function(){return '';}; // row class handler
        var renderedRowsAllowed = this.renderedRowsAllowed;
        var renderedRows = this.renderedRows;
        var cellHeight = this.cellHeight;
        var iCellHeight = cellHeight - 6;
        var headerWidth = this.headerWidth;
        var cm = this.columnModel;
        var fontSize = this.fontSize;
        var gap = ((this.gap == 0)? 2 : 0);

        var html = [];
        var idx = 0;

        var firstRenderingFlg = false;
        if (renderedRows == 0) firstRenderingFlg = true;

        if (firstRenderingFlg) {
            this.innerBodyDiv.setStyle({height: (rows.length * cellHeight) + 'px'});
            html[idx++] = '<table id="mtgBT'+id+'" border="0" cellspacing="0" cellpadding="0" width="'+headerWidth+'" class="mtgBodyTable">';
            html[idx++] = '<tbody>';
        }
        var lastRowToRender = renderedRowsAllowed + renderedRows;
        if (lastRowToRender > rows.length) lastRowToRender = rows.length;
        $('mtgLoader'+id).show();
        for (var i = renderedRows; i < lastRowToRender; i++) {
            var row = rows[i];
            html[idx++] = '<tr id="mtgRow'+id+'_'+i+'" class="mtgRow'+id+' '+rc(i)+'" style="'+rs(i)+'">';
            for (var j = 0; j < cm.length; j++) {
                var columnIdx = cm[j].positionIndex;
                var type = cm[j].type || 'string';
                var cellWidth = parseInt(cm[j].width);
                var iCellWidth = cellWidth - 6 - gap;
                var editor = cm[j].editor || null;
                var normalEditorFlg = !(editor == 'checkbox' || editor instanceof MyTableGrid.CellCheckbox || editor == 'radio' || editor instanceof MyTableGrid.CellRadioButton);
                var alignment = 'left';
                if (!cm[j].hasOwnProperty('renderer')) {
                    alignment = (type == 'string')? 'left' : 'right';
                }
                if (cm[j].hasOwnProperty('align')) {
                    alignment = cm[j].align;
                }

                var temp = tdTmpl.replace(/\{id\}/g, id);
                temp = temp.replace(/\{x\}/g, j);
                temp = temp.replace(/\{y\}/g, i);
                temp = temp.replace(/\{width\}/g, cellWidth);
                temp = temp.replace(/\{height\}/g, cellHeight);

                html[idx++] = temp;

                temp = icTmpl.replace(/\{id\}/g, id);
                temp = temp.replace(/\{x\}/g, j);
                temp = temp.replace(/\{y\}/g, i);
                temp = temp.replace(/\{width\}/, iCellWidth);
                temp = temp.replace(/\{height\}/, iCellHeight);
                temp = temp.replace(/\{align\}/, alignment);

                html[idx++] = temp;
                if (normalEditorFlg) { // checkbox is an special case
                    if (!cm[j].hasOwnProperty('renderer')) {
                        html[idx++] = this.getStringMask(row[columnIdx], cellWidth, fontSize);
                    } else {
                        html[idx++] = cm[j].renderer(row[columnIdx]);
                    }
                } else if (editor == 'checkbox' || editor instanceof MyTableGrid.CellCheckbox) {
                    temp = checkboxTmpl.replace(/\{id\}/g, id);
                    temp = temp.replace(/\{x\}/g, j);
                    temp = temp.replace(/\{y\}/g, i);
                    temp = temp.replace(/\{value\}/, row[columnIdx]);
                    html[idx++] = temp;
                } else if (editor == 'radio' || editor instanceof MyTableGrid.CellRadioButton) {
                    temp = radioTmpl.replace(/\{id\}/g, id);
                    temp = temp.replace(/\{x\}/g, j);
                    temp = temp.replace(/\{y\}/g, i);
                    temp = temp.replace(/\{value\}/, row[columnIdx]);
                    html[idx++] = temp;
                }
                html[idx++] = '</div>';
                html[idx++] = '</td>';
            }
            html[idx++] = '</tr>';
            renderedRows++;
        }
        if (firstRenderingFlg) {
            html[idx++] = '</tbody>';
            html[idx++] = '</table>';
        }
        this.renderedRows = renderedRows;
        setTimeout(function(){$('mtgLoader'+id).hide();},1.5); //just to see the spiner 
        return html.join('');
    },

    /**
     * Applies cell callbacks
     */
    applyCellCallbacks : function() {
        for (var i = 0; i < this.columnModel.length; i++) {
            var editor = this.columnModel[i].editor;
            if ((editor instanceof MyTableGrid.CellRadioButton && editor.onClickCallback) ||
                (editor instanceof MyTableGrid.CellCheckbox && editor.onClickCallback)) {
                $$('.mtgInput'+this.mtgId + '_' + i).each(function(element){
                    element.onclick = (function(editor) {
                        return function() {
                            editor.onClickCallback(element.value, true);
                        };
                    })(editor);
                });
            }
        }
    },

    /**
     * Returns an string mask when the string is larger than the containing cell width
     */
    getStringMask : function(value, width, fontSize) {
        var result = value;
        if (Math.round(value.length*fontSize*0.6) > width) {
            var nbrChar = Math.round((width / fontSize)*1.5);
            result = value.substring(0, nbrChar - 3) + ' ...';
        }
        return result;
    },

    /**
     * Creates the Setting Menu
     */
    createSettingMenu : function() {
        var id = this.mtgId;
        var cm = this.columnModel;
        var bh = this.bodyHeight + 30;
        var cellHeight = (Prototype.Browser.IE)? 25 : 22;
        var height = (cm.length * 25 > bh)? bh : cm.length * cellHeight;
        var html = [];
        var idx = 0;
        html[idx++] = '<div id="mtgSM'+id+'" class="mtgMenu" style="height:'+height+'px">';
        html[idx++] = '<ul>';
        for (var i = 0; i < cm.length; i++) {
            var column = cm[i];
            html[idx++] = '<li>';
            html[idx++] = '<a href="#" class="mtgMenuItem">';
            html[idx++] = '<table border="0" cellpadding="0" cellspacing="0" width="100%">';
            html[idx++] = '<tr><td width="25"><span><input type="checkbox" checked="'+column.visible+'"></span></td>';
            html[idx++] = '<td>&nbsp;'+ column.title+'</td></tr>';
            html[idx++] = '</table>';
            html[idx++] = '</a>';
            html[idx++] = '</li>';
        }
        html[idx++] = '</ul>';
        html[idx++] = '</div>';
        return html.join('');
    },

    /**
     * Applies Setting Menu behavior
     */
    applySettingMenuBehavior : function() {
        var settingMenu = $('mtgSM' + this.mtgId);
        var width = settingMenu.getWidth();
        var settingButton = $('mtgSB' + this.mtgId);

        Event.observe(settingButton, 'click', function() {
            if (settingMenu.getStyle('visibility') == 'hidden') {
                var topPos = settingButton.offsetTop;
                var leftPos = settingButton.offsetLeft;
                settingMenu.setStyle({
                    top: (topPos + 16) + 'px',
                    left: (leftPos - width + 16) + 'px',
                    visibility: 'visible'
                });
            } else {
                settingMenu.setStyle({visibility: 'hidden'});
            }
        });

        Event.observe($('mtgHeaderTitle'+this.mtgId),'mousemove', function() {
           settingMenu.setStyle({visibility: 'hidden'});
        });

        var self = this;
        $$('#mtgSM' + this.mtgId + ' input[@type:checkbox]').each(function(checkbox, index) {
            checkbox.onclick = function() {
                self.toggleColumnVisibility(index, checkbox.checked);
            };
        });
    },

    /**
     * Synchronizes horizontal scrolling
     */
    syncScroll : function() {
        var id = this.mtgId;
        var keys = this.keys;
        var bodyDiv = this.bodyDiv;
        var headerRowDiv = this.headerRowDiv;
        var bodyTable = this.bodyTable;
        var renderedRows = this.renderedRows;
        var cellHeight = this.cellHeight;

        this.scrollLeft = headerRowDiv.scrollLeft = bodyDiv.scrollLeft;
        this.scrollTop = bodyDiv.scrollTop;

        $('mtgHB' + id).setStyle({visibility: 'hidden'});
        $('mtgHBM' + id).setStyle({visibility: 'hidden'});

        if(renderedRows < this.rows.length 
            && (bodyTable.getHeight() - bodyDiv.scrollTop - 10) < bodyDiv.clientHeight) {
            var html = this.createTableBody(this.rows);
            bodyTable.down('tbody').insert(html);
            this.addKeyBehavior();
            keys.addMouseBehavior();
        }
    },

    /**
     * Makes all columns resizable
     */
    makeAllColumnResizable: function() {
        var id = this.mtgId;
        var topPos = 0;
        if (this.options.title) topPos += this.headerHeight;
        var headerButton = $('mtgHB' + this.mtgId);
        var columnIndex;
        var self = this;
        $$('.mtgHS' + this.mtgId).each(function(separator, index) {
            Event.observe(separator, 'mousemove', function() {
                headerButton.setStyle({visibility: 'visible'});
                columnIndex = -1;
                for (var i = 0; i < self.columnModel.length; i++) {
                    if (index == self.columnModel[i].positionIndex) {
                        columnIndex = i;
                        break;
                    }
                }
                if (columnIndex >= 0) {
                    var markerHeight = self.bodyHeight + self.headerHeight + 2;
                    if (self.hasHScrollBar()) markerHeight = markerHeight - self.scrollBarWidth;

                    var leftPos = $('mtgHC' + id + '_' + columnIndex).offsetLeft - self.scrollLeft;

                    self.resizeMarkerLeft.setStyle({
                        height: markerHeight + 'px',
                        top: (topPos + 2) + 'px',
                        left: leftPos + 'px'
                    });
                    leftPos += $('mtgHC' + id + '_' + columnIndex).offsetWidth;

                    self.resizeMarkerRight.setStyle({
                        height: markerHeight + 'px',
                        top: (topPos + 2) + 'px',
                        left: (leftPos + 1) + 'px'
                    });
                }
            });
        });

        new Draggable(self.resizeMarkerRight, {
            constraint: 'horizontal',
            onStart : function() {
                self.resizeMarkerRight.setStyle({
                    backgroundColor: 'dimgray'
                });

                self.resizeMarkerLeft.setStyle({
                    backgroundColor: 'dimgray'
                });
            },

            onEnd : function() {
                var newWidth = parseInt(self.resizeMarkerRight.getStyle('left')) - parseInt(self.resizeMarkerLeft.getStyle('left'));
                if (newWidth > 0 && columnIndex != null) {
                    setTimeout(function() {
                        self.resizeColumn(columnIndex, newWidth);
                    }, 0);
                }

                self.resizeMarkerLeft.setStyle({
                    backgroundColor: 'transparent',
                    left: 0
                });

                self.resizeMarkerRight.setStyle({
                    backgroundColor: 'transparent'
                });
            },
            endeffect : false
        });
    },

    /**
     * Resizes a column to a new size
     *
     * @param index the index column position
     * @param newWidth resizing width
     */
    resizeColumn: function(index, newWidth) {
        var oldWidth = parseInt($('mtgHC' + this.mtgId + '_' + index).width);
        var editor = this.columnModel[index].editor;
        var checkboxOrRadioFlg = (editor == 'checkbox' || editor instanceof MyTableGrid.CellCheckbox || editor == 'radio' || editor instanceof MyTableGrid.CellRadioButton);

        $('mtgHC' + this.mtgId + '_' + index).width = newWidth;
        $('mtgHC' + this.mtgId + '_' + index).setStyle({width: newWidth + 'px'});

        $('mtgIHC' + this.mtgId + '_' + index).setStyle({width: (newWidth - 8 - ((this.gap == 0) ? 2 : 0)) + 'px'});

        var self = this;

        $$('.mtgC' + this.mtgId + '_' + index).each(function(cell) {
            cell.width = newWidth;
            cell.setStyle({width: newWidth + 'px'});
        });


        $$('.mtgIC' + this.mtgId + '_' + index).each(function(cell, y) {
            var value = self.getValueAt(index, y);
            cell.setStyle({width: (newWidth - 6 - ((self.gap == 0) ? 2 : 0)) + 'px'});
            if (!checkboxOrRadioFlg) {
                if (self.columnModel[index].renderer) value = self.columnModel[index].renderer(value);
                if (!value.match(/^<img/)) {
                    cell.innerHTML = self.getStringMask(value, newWidth, self.fontSize);
                } else {
                    cell.innerHTML = value;
                }
            }
        });

        this.headerWidth = this.headerWidth - (oldWidth - newWidth);

        $('mtgHRT' + this.mtgId).width = this.headerWidth;
        $('mtgBT' + this.mtgId).width = this.headerWidth;

        this.columnModel[index].width = newWidth;
        this.syncScroll();
    },

    hasHScrollBar : function() {
        return (this.headerWidth + 20) > this.tableWidth;
    },

    /**
     * Makes all columns draggable
     */
    makeAllColumnDraggable : function() {
        this.separators = [];
        var i = 0;
        var id = this.mtgId;
        var self = this;
        $$('.mtgHS' + this.mtgId).each(function(separator) {
            self.separators[i++] = separator;
        });

        var topPos = 0;
        if (this.options.title) topPos += this.headerHeight;

        var dragColumn = $('dragColumn' + id);

        $$('.mtgIHC' + id).each(function(column, index) {
            var columnIndex = -1;
            Event.observe(column, 'mousemove', function() {
                var leftPos = column.up().offsetLeft;
                dragColumn.setStyle({
                    top: (topPos + 15) + 'px',
                    left: (leftPos - self.scrollLeft + 15) + 'px'
                });
            });
            var dragger = new Draggable(dragColumn, {
                handle : column,
                onStart : function() {
                    for (var i = 0; i < self.columnModel.length; i++) {
                        if (index == self.columnModel[i].positionIndex) {
                            columnIndex = i;
                            break;
                        }
                    }
                    if (Prototype.Browser.IE) {
                        // The drag might register an ondrag or onselectstart event when using IE
                        Event.observe(document.body, "drag", function() {return false;}, false);
                        Event.observe(document.body, "selectstart",	function() {return false;}, false);
                    }
                    dragColumn.down('span').innerHTML = self.columnModel[columnIndex].title;
                    dragColumn.setStyle({visibility: 'visible'});
                },
                onDrag : function() {
                    var leftPos = parseInt(dragColumn.getStyle('left'));
                    var width = parseInt(dragColumn.getStyle('width'));
                    setTimeout(function(){
                        self.detectDroppablePosition(leftPos + width / 2, width, dragColumn, columnIndex);
                    }, 0);
                },
                onEnd : function() {
                    dragColumn.setStyle({visibility: 'hidden'});
                    self.colMoveTopDiv.setStyle({visibility: 'hidden'});
                    self.colMoveBottomDiv.setStyle({visibility: 'hidden'});
                    if (columnIndex >=0 && self.targetColumnId >= 0) {
                        setTimeout(function(){
                            self.moveColumn(columnIndex, self.targetColumnId);
                            columnIndex = -1;
                        }, 0);
                    }
                },
                endeffect : false
            });
        });
    },

    /**
     * Detects droppable position when the mouse pointer is over a header cell 
     * separator
     */
    detectDroppablePosition : function(columnPos, width, dragColumn, index) {
        var topPos = -8;
        if (this.options.title) topPos += this.headerHeight;
        var sepLeftPos = 0;
        var cm = this.columnModel;
        var gap = this.gap;
        var scrollLeft = this.scrollLeft;
        var colMoveTopDiv = this.colMoveTopDiv;
        var colMoveBottomDiv = this.colMoveBottomDiv;

        for (var i = 0; i < cm.length; i++) {
            if (cm[i].visible) sepLeftPos += parseInt(cm[i].width) + gap;
            if (columnPos > (sepLeftPos - scrollLeft)
                    && (columnPos - (sepLeftPos - this.scrollLeft)) < (width / 2)) {
                colMoveTopDiv.setStyle({
                    top: topPos + 'px',
                    left: (sepLeftPos - scrollLeft - 4) + 'px',
                    visibility : 'visible'
                });
                colMoveBottomDiv.setStyle({
                    top: (topPos + 34) + 'px',
                    left: (sepLeftPos - scrollLeft - 4) + 'px',
                    visibility : 'visible'
                });
                this.targetColumnId = i;
                var className = (i != index)? 'drop-yes' : 'drop-no';
                dragColumn.down('div').className = className;
                break;
            } else {
                colMoveTopDiv.setStyle({visibility : 'hidden'});
                colMoveBottomDiv.setStyle({visibility : 'hidden'});
                this.targetColumnId = null;
                dragColumn.down('div').className = 'drop-no';
            }
        }
    },

    /**
     * Moves a column from one position to a new one
     *
     * @param fromColumnId initial position
     * @param toColumnId target position
     */
    moveColumn : function(fromColumnId, toColumnId) {
        // Some validations
        if (fromColumnId == null
            || toColumnId == null
            || fromColumnId == toColumnId
            || (toColumnId + 1 == fromColumnId && fromColumnId == this.columnModel.length -1)) return;

        var id = this.mtgId;
        var cm = this.columnModel;
        var gap = (this.gap == 0) ? 2 : 0;
        var keys = this.keys;
        var rows = this.rows;
        var renderedRows = this.renderedRows;
        var self = this;
        $('mtgHB' + id).setStyle({visibility: 'hidden'}); // in case the cell menu button is visible
        this.blurCellElement(keys._nCurrentFocus); //in case there is a cell in editing mode
        keys.blur(); //remove the focus of the selected cell

        // Moving header cell element
        if (toColumnId == 0) {
            var removedHeaderCell = $('mtgHC'+id+'_'+fromColumnId).remove();
            var targetHeaderCell = $('mtgHC'+id+'_'+ toColumnId);
            targetHeaderCell.up().insertBefore(removedHeaderCell, targetHeaderCell);

            // Moving cell elements
            var removedCells = [];
            var idx = 0;
            $$('.mtgC'+id+'_'+fromColumnId).each(function(element){
                removedCells[idx++] = element.remove();
            });

            for (var i = 0; i < renderedRows; i++) {
                var targetCell = $('mtgC'+id+'_'+toColumnId+','+i);
                targetCell.up().insertBefore(removedCells[i], targetCell);
            }
        } else if (toColumnId > fromColumnId && toColumnId < cm.length - 1) {
            var removedHeaderCell = $('mtgHC'+id+'_'+fromColumnId).remove();
            var targetId = toColumnId + 1;
            var targetHeaderCell = $('mtgHC'+id+'_'+ targetId);
            targetHeaderCell.up().insertBefore(removedHeaderCell, targetHeaderCell);

            // Moving cell elements
            var removedCells = [];
            var idx = 0;
            $$('.mtgC'+id+'_'+fromColumnId).each(function(element){
                removedCells[idx++] = element.remove();
            });

            for (var i = 0; i < renderedRows; i++) {
                var targetCell = $('mtgC'+id+'_'+targetId+','+i);
                targetCell.up().insertBefore(removedCells[i], targetCell);
            }
        } else if (toColumnId  < fromColumnId) {
            var removedHeaderCell = $('mtgHC'+id+'_'+fromColumnId).remove();
            var targetId = toColumnId + 1;
            var targetHeaderCell = $('mtgHC'+id+'_'+ targetId);
            targetHeaderCell.up().insertBefore(removedHeaderCell, targetHeaderCell);

            // Moving cell elements
            var removedCells = [];
            var idx = 0;
            $$('.mtgC'+id+'_'+fromColumnId).each(function(element){
                removedCells[idx++] = element.remove();
            });

            for (var i = 0; i < renderedRows; i++) {
                var targetCell = $('mtgC'+id+'_'+targetId+','+i);
                targetCell.up().insertBefore(removedCells[i], targetCell);
            }
        } else if (toColumnId == cm.length - 1) { // The last cell
            var tr = $('mtgHC'+id+'_'+fromColumnId).up();
            var removedHeaderCell = $('mtgHC'+id+'_'+fromColumnId).remove();
            var last = $('mtgHC'+id+'_'+ cm.length);
            tr.insertBefore(removedHeaderCell, last);

            // Moving cell elements
            var removedCells = [];
            var idx = 0;
            $$('.mtgC'+id+'_'+fromColumnId).each(function(element){
                removedCells[idx++] = element.remove();
            });

            for (var i = 0; i < renderedRows; i++) {
                var tr = $('mtgRow'+id+'_'+i);
                tr.insert(removedCells[i]);
            }
        }

        // Update column model
        var columnModelLength = cm.length;
        var columnModelEntry = cm[fromColumnId];
        cm[fromColumnId] = null;
        cm = cm.compact();
        var aTemp = [];
        var k = 0;
        var targetColumnId = toColumnId;
        if (toColumnId > 0 && toColumnId < fromColumnId) targetColumnId++;
        for (var c = 0; c < columnModelLength; c++) {
            if (c == targetColumnId) aTemp[k++] = columnModelEntry;
            if (c < (columnModelLength - 1))
                aTemp[k++] = cm[c];
        }
        cm = this.columnModel = aTemp;
        var htr = $('mtgHRT'+id).down('tr');
        htr.getElementsBySelector('th').each(function(th, index){
            if (index < cm.length) {
                th.id = 'mtgHC'+id+'_'+index;
                var ihc = th.down('div');
                ihc.id = 'mtgIHC'+id+'_'+index;
                ihc.down('img').id = 'mtgSortIcon'+id+'_'+index;
                ihc.down('div').id = 'mtgHS'+id+'_'+index;
            }
        });

        // Recreates cell indexes
        for (var i = 0; i < renderedRows; i++) {
            $$('.mtgR'+id+'_'+i).each(function(td, index) {
                td.id = 'mtgC'+id+'_'+index+','+i;
                td.className = 'mtgCell mtgC'+id+' mtgC'+id+'_'+index + ' mtgR'+id+'_'+i;
            });

            $$('.mtgIR'+id+'_'+i).each(function(div, index) {
                div.id = 'mtgIC'+id+'_'+index+','+i;
                div.className = 'mtgInnerCell mtgIC'+id+' mtgIC'+id+'_'+index + ' mtgIR'+id+'_'+i;
                if (div.firstChild.tagName == 'INPUT') {
                    var input = div.firstChild;
                    input.id = 'mtgInput'+id+'_'+index+','+i
                    input.name = 'mtgInput'+id+'_'+index+','+i
                    input.className =  input.className.replace(/mtgInput.*?_.*?\s/, 'mtgInput'+id+'_'+index+' ');
                }
            });
        }
        this.addKeyBehavior();
    },

    /**
     * Add Key behavior functionality to the table grid
     */
    addKeyBehavior : function() {
        var self = this;
        var id = this.mtgId;
        var rows = this.rows;
        var cm = this.columnModel;
        var keys = this.keys;
        var renderedRows = this.renderedRows;
        var renderedRowsAllowed = this.renderedRowsAllowed;
        var beginAtRow = renderedRows - renderedRowsAllowed;
        if (beginAtRow < 0) beginAtRow = 0;
        for (var j = beginAtRow; j < renderedRows; j++) {
            for (var i = 0; i < cm.length; i++) {
                var element = $('mtgC'+id+'_'+i+','+j);
                if (cm[i].editable) {
                    keys.event.remove.action(element);
                    keys.event.remove.esc(element);
                    keys.event.remove.blur(element);                

                    var f_action = (function(element) {
                        return function() {
                            if (self.editedCellId == null || self.editedCellId != element.id) {
                                self.editedCellId = element.id;
                                self.editCellElement(element);
                            } else {
                                self.blurCellElement(element);
                                self.editedCellId = null;
                            }
                        };
                    })(element);
                    keys.event.action(element, f_action);

                    var f_esc = (function(element) {
                        return function() {
                            self.blurCellElement(element);
                            self.editedCellId = null;
                        };
                    })(element);
                    keys.event.esc(element, f_esc);

                    var f_blur = (function(x, y, element) {
                        return function() {
                            self.blurCellElement(element);
                            self.editedCellId = null;
                            if (self.onCellBlur) self.onCellBlur(element, rows[y][x], x, y, cm[x].id);
                        };
                    })(i, j, element);
                    keys.event.blur(element, f_blur);
                }

                keys.event.remove.focus(element);
                var f_focus = (function(x, y, element) {
                    return function() {
                        if (self.onCellFocus) {
                            self.onCellFocus(element, rows[y][x], x, y, cm[x].id);
                        }
                    };
                })(i, j, element);
                keys.event.focus(element, f_focus);
            }
        }
    },

    /**
     *  When a cell is edited
     */
    editCellElement : function(element) {
        this.keys._bInputFocused = true;
        var width = parseInt(element.getStyle('width'));
        var height = parseInt(element.getStyle('height'));
        var coords = this.getCurrentPosition();
        var x = coords[0];
        var y = coords[1];
        var id = 'mtgIC' + this.mtgId + '_' + x +','+y;
        var innerElement = $(id);
        var value = this.getValueAt(x, y);
        var editor = this.columnModel[x].editor || 'input';
        var type = this.columnModel[x].type || 'string';
        var marginTop = (Prototype.Browser.IE)? '1px' : '2px';
        var input = null;
        var isInputFlg = (editor == 'input' || editor instanceof MyTableGrid.ComboBox || editor instanceof MyTableGrid.BrowseInput || editor instanceof MyTableGrid.CellCalendar);

        if (isInputFlg) {
            element.setStyle({
                height: this.cellHeight + 'px'
            });

            innerElement.setStyle({
                width: width + 'px',
                height: height + 'px',
                padding: '0',
                border: '0',
                verticalAlign: 'middle',
                textAlign: 'left'
            });

            var alignment = (type == 'number')? 'right' : 'left';

            if (editor == 'input') {
                innerElement.innerHTML = '';
                input = new Element('input', {id: 'mtgInput' + this.mtgId + '_' + x + ',' + y,
                    type: 'text',
                    value: value
                });
                input.addClassName('mtgInputText');
                input.setStyle({
                    width: (width - 9) + 'px',
                    height: (height - 10) + 'px',
                    marginTop: marginTop,
                    marginLeft: '1px',
                    paddingLeft: '3px',
                    paddingRight: '3px',
                    textAlign: alignment
                });
                innerElement.appendChild(input);
                input.focus();
                input.select();
            } else {
                value = innerElement.innerHTML;
                innerElement.innerHTML = '';
                input = editor.render(this,{width: width, height: height, value: value, align: alignment});
                innerElement.appendChild(input);
                input.down('input').focus();
                input.down('input').select();
            } 
        } else if (editor == 'checkbox' || editor instanceof MyTableGrid.CellCheckbox) {
            input = $('mtgInput' + this.mtgId + '_' + x + ',' + y);
            input.checked = (!input.checked);
            if (editor instanceof MyTableGrid.CellCheckbox && editor.onClickCallback) {
                editor.onClickCallback(value, input.checked);
            }
            this.keys._bInputFocused = false;
            this.editedCellId = null;
        } else if (editor == 'radio' || editor instanceof MyTableGrid.CellRadioButton) {
            input = $('mtgInput' + this.mtgId + '_' + x + ',' + y);
            input.checked = (!input.checked);
            if (editor instanceof MyTableGrid.CellRadioButton && editor.onClickCallback) {
                editor.onClickCallback(value, input.checked);
            }
            this.keys._bInputFocused = false;
            this.editedCellId = null;
        }
    },
    /**
     * When the cell is blured
     */
    blurCellElement : function(element) {
        if (!this.keys._bInputFocused) return;
        var self = this;
        var id = this.mtgId;
        var keys = this.keys;
        var cm = this.columnModel;
        var fs = this.fontSize;
        var width = parseInt(element.getStyle('width'));
        var height = parseInt(element.getStyle('height'));
        var coords = keys.getCoordsFromCell(element);
        var x = coords[0];
        var y = coords[1];
        var cellWidth = cm[x].width;
        var cellHeight = this.cellHeight;

        var innerId = 'mtgIC' + id + '_' + x + ',' + y;
        var input = $('mtgInput' + id + '_' + x + ',' + y);
        var value = input.value;
        var innerElement = $(innerId);
        var editor = cm[x].editor || 'input';
        var type = cm[x].type || 'string';
        var alignment = (type == 'number')? 'right' : 'left';
        var isInputFlg = (editor == 'input' || editor instanceof MyTableGrid.ComboBox || editor instanceof MyTableGrid.BrowseInput || editor instanceof MyTableGrid.CellCalendar);

        if (isInputFlg) {
            if (editor.hide) editor.hide(); // this only when the ComboBox editor is used
            if (editor.validate) { // this only when the ComboBox editor is used
                if (!editor.validate(value, input)) {
                    value = editor.getList()[0].text;
                }
            } 
            
            element.setStyle({
                height: cellHeight + 'px'
            });

            innerElement.setStyle({
                width : (width - 6) + 'px',
                height : (height - 6) + 'px',
                padding: '3px',
                textAlign: alignment
            }).update(this.getStringMask(value, cellWidth, fs));
        }

        x = this.getColumnIndex(this.columnModel[x].id);
        this.rows[y][x] = value;

        if (editor instanceof MyTableGrid.BrowseInput && editor.afterUpdateCallback) {
            editor.afterUpdateCallback(element, value);
        }

        if (this.modifiedRows.indexOf(y) == -1) this.modifiedRows.push(y); //if doesn't exist in the array the row is registered
        keys._bInputFocused = false;
    },

    /**
     * Applies header buttons
     */
    applyHeaderButtons : function() {
        var self = this;
        var id = this.mtgId;
        var headerButton = $('mtgHB' + this.mtgId);
        var headerButtonMenu = $('mtgHBM' + this.mtgId);
        var sortAscMenuItem = $('mtgSortAsc'+this.mtgId);
        var sortDescMenuItem = $('mtgSortDesc'+this.mtgId);
        var topPos = 0;
        if (this.options.title) topPos += this.headerHeight;
        var selectedHCIndex = -1;
        $$('.mtgIHC' + id).each(function(element, index) {
            Event.observe(element, 'mousemove', function() {
                for (var i = 0; i < self.columnModel.length; i++) {
                    if (index == self.columnModel[i].positionIndex) {
                        selectedHCIndex = i;
                        break;
                    }
                }
                if (selectedHCIndex == -1) return;
                headerButtonMenu.setStyle({visibility: 'hidden'});
                var hc = element.up();
                var leftPos = hc.offsetLeft + hc.offsetWidth;
                leftPos = leftPos - 16 - self.scrollLeft;
                if (leftPos < self.bodyDiv.clientWidth) {
                    headerButton.setStyle({
                        top: (topPos + 2) + 'px',
                        left: leftPos + 'px',
                        visibility: 'visible'
                    });
                }
                sortAscMenuItem.onclick = function() {
                    $('mtgSortIcon'+id+'_'+selectedHCIndex).src = self.imageRefs.mtgSortArrowAscIcon;
                    self.request[self.sortColumnParameter] = self.columnModel[selectedHCIndex].id;
                    self.request[self.ascDescFlagParameter] = 'ASC';
                    self.retrieveDataFromUrl(1);
                    $('mtgSortIcon'+id+'_'+self.sortedColumnIndex).hide();
                    $('mtgSortIcon'+id+'_'+selectedHCIndex).show();
                    self.sortedColumnIndex = selectedHCIndex;
                };

                sortDescMenuItem.onclick = function() {
                    $('mtgSortIcon'+id+'_'+selectedHCIndex).src = self.imageRefs.mtgSortArrowDescIcon;
                    self.request[self.sortColumnParameter] = self.columnModel[selectedHCIndex].id;
                    self.request[self.ascDescFlagParameter] = 'DESC';
                    self.retrieveDataFromUrl(1);
                    $('mtgSortIcon'+id+'_'+self.sortedColumnIndex).hide();
                    $('mtgSortIcon'+id+'_'+selectedHCIndex).show();
                    self.sortedColumnIndex = selectedHCIndex;
                };
            });
        });

        Event.observe(headerButton, 'click', function() {
            if (headerButtonMenu.getStyle('visibility') == 'hidden') {
                var selectAllItem = $$('#mtgHBM' + id + ' .mtgSelectAll')[0];
                if (self.columnModel[selectedHCIndex].editor == 'checkbox' ||
                    self.columnModel[selectedHCIndex].editor instanceof MyTableGrid.CellCheckbox) {
                    selectAllItem.show();
                    var index = self.columnModel[selectedHCIndex].positionIndex;
                    selectAllItem.onclick = function() {
                        var flag = $('mtgSelectAll' + id).checked;
                        $$('.mtgInput' + id + '_' + selectedHCIndex).each(function(element) {
                            element.checked = flag;
                        });

                        if (self.columnModel[selectedHCIndex].editor instanceof MyTableGrid.CellCheckbox
                                && self.columnModel[selectedHCIndex].editor.onClickCallback) {
                            self.columnModel[selectedHCIndex].editor.onClickCallback();
                        }
                    };
                } else {
                    selectAllItem.hide();
                }

                var leftPos = parseInt(headerButton.getStyle('left'));
                var topPos = self.headerHeight + 2;
                if (self.options.title) topPos += self.headerHeight;
                headerButtonMenu.setStyle({
                    top: topPos + 'px',
                    left: leftPos + 'px',
                    visibility: 'visible'
                });
            } else {
                headerButtonMenu.setStyle({visibility: 'hidden'});
            }
        });
    },

    toggleColumnVisibility : function(index, visibleFlg) {
        this.blurCellElement(this.keys._nCurrentFocus); //in case there is a cell in editing mode
        this.keys.blur(); //remove the focus of the selected cell
        var headerRowTable = $('mtgHRT' + this.mtgId);
        var bodyTable = $('mtgBT' + this.mtgId);

        for (var i = 0; i < this.columnModel.length; i++) {
            if (this.columnModel[i].positionIndex == index) {
                index = i;
                break;
            }
        }

        var targetColumn = $('mtgHC' + this.mtgId + '_' + index);
        $('mtgHB' + this.mtgId).setStyle({visibility: 'hidden'});

        var width = 0;

        if (!visibleFlg) { // hide
            width = parseInt(targetColumn.offsetWidth);
            targetColumn.hide();
            $$('.mtgC'+this.mtgId+ '_'+index).each(function(element){
                element.hide();
            });
            this.columnModel[index].visible = false;
            this.headerWidth = this.headerWidth - width;
        } else { // show
            targetColumn.show();
            width = parseInt(targetColumn.offsetWidth);
            $$('.mtgC'+this.mtgId+ '_'+index).each(function(element){
                element.show();
            });
            this.columnModel[index].visible = true;
            this.headerWidth = this.headerWidth + width;
        }

        headerRowTable.width = this.headerWidth;
        headerRowTable.setStyle({width: this.headerWidth + 'px'});
        bodyTable.width = this.headerWidth;
        bodyTable.setStyle({width: this.headerWidth + 'px'});
    },

    fullPadding : function(element, s) {
        var padding = parseInt(element.getStyle('padding-'+s));
        padding = (isNaN(padding)) ? 0 : padding;
        var border = parseInt(element.getStyle('border-'+s+'-width'));
        border = (isNaN(border)) ? 0 : border;
        return padding + border;
    },

    retrieveDataFromUrl : function(pageNumber, firstTimeFlg) {
        var pageParameter = this.pager.pageParameter || 'page';
        this.request[pageParameter] = pageNumber;
        $('mtgLoader'+this.mtgId).show();
        var self = this;
        new Ajax.Request(this.url, {
            parameters: self.request,
            onSuccess: function(transport) {
                var tableModel = transport.responseText.evalJSON();
                self.rows = tableModel.rows || [];
                self.pager = tableModel.options.pager || null;
                if (tableModel.rows.length > 0) {
                    self.renderedRows = 0;
                    self.innerBodyDiv.innerHTML = self.createTableBody(tableModel.rows);
                    self.bodyTable = $('mtgBT' + self.mtgId);
                    if (!firstTimeFlg) {
                        self.applyCellCallbacks();
                        self.keys = new KeyTable(self);
                        self.addKeyBehavior();
                    }
                }
                self.pagerDiv.innerHTML = self.updatePagerInfo(); // update pager info panel
                self.addPagerBehavior();

                if (self.afterRender) {
                    self.afterRender();
                }
                $('mtgLoader'+self.mtgId).hide();
                if (firstTimeFlg) self.bodyDiv.fire('dom:dataLoaded');
            }
        });
    },

    updatePagerInfo : function(emptyFlg) {
        var id = this.mtgId;
        var imageRefs = this.imageRefs;

        if (emptyFlg)
            return '<span id=\"mtgLoader'+id+'\" style=\"display:none\"><img src=\"'+imageRefs.mtgLoaderIcon+'\" width=\"16\" height=\"16\"></span>';

        var html = [];
        var idx = 0;
        var pager = this.pager;

        if (this.pager.total > 0) {
            var temp = this.messages.totalDisplayMsg;
            temp = temp.replace(/\{total\}/g, pager.total);
            if (pager.from && pager.to) {
                temp += this.messages.rowsDisplayMsg;
                temp = temp.replace(/\{from\}/g, pager.from);
                temp = temp.replace(/\{to\}/g, pager.to);
            }
            html[idx++] = '<span class=\"mtgPagerMsg\">'+temp+'</span>';
            if (pager.pages) {
                temp = this.messages.pagePromptMsg;
                temp = temp.replace(/\{pages\}/g, pager.pages);
                var input = '<input type="text" name="mtgPageInput'+id+'" id="mtgPageInput'+id+'" value="'+pager.currentPage+'" class="mtgPageInput" size="3" maxlength="3">';
                temp = temp.replace(/\{input\}/g, input);
                html[idx++] = '<table class=\"mtgPagerTable\" border=\"0\" cellpadding=\"0\" cellspacing=\"0\">';
                html[idx++] = '<tbody>';
                html[idx++] = '<tr>';

                html[idx++] = '<td><span id=\"mtgLoader'+id+'\" style=\"display:none\"><img src=\"'+imageRefs.mtgLoaderIcon+'\" width=\"16\" height=\"16\"></span></td>';
                html[idx++] = '<td><img src=\"'+imageRefs.ySepIcon+'\" width=\"2\" height=\"13\"></td>';
                html[idx++] = '<td><a id=\"mtgFirst'+id+'\" class=\"mtgPagerCtrl\"><img src=\"'+imageRefs.firstIcon+'\" width=\"16\" height=\"16\" border=\"0\"></a></td>';
                html[idx++] = '<td><a id=\"mtgPrev'+id+'\" class=\"mtgPagerCtrl\"><img src=\"'+imageRefs.prevIcon+'\" width=\"16\" height=\"16\" border=\"0\"></a></td>';
                html[idx++] = '<td><img src=\"'+imageRefs.ySepIcon+'\" width=\"2\" height=\"13\"></td>';
                html[idx++] = temp;

                html[idx++] = '<td><img src=\"'+imageRefs.ySepIcon+'\" width=\"2\" height=\"13\"></td>';
                html[idx++] = '<td><a id=\"mtgNext'+id+'\" class=\"mtgPagerCtrl\"><img src=\"'+imageRefs.nextIcon+'\" width=\"16\" height=\"16\" border=\"0\"></a></td>';
                html[idx++] = '<td><a id=\"mtgLast'+id+'\" class=\"mtgPagerCtrl\"><img src=\"'+imageRefs.lastIcon+'\" width=\"16\" height=\"16\" border=\"0\"></a></td>';
                html[idx++] = '</tr>';
                html[idx++] = '</table>';
            }
        } else {
            html[idx++] = '<span class=\"mtgPagerMsg\">'+this.messages.pagerNoDataFound+'</span>';
        }
        return html.join('')
    },

    addPagerBehavior : function() {
        var self = this;
        if (!self.pager.pages) return;
        var currentPage = self.pager.currentPage;
        var pages = self.pager.pages;
        var total = self.pager.total;
        if (total > 0) {
            if (currentPage > 1) {
                $('mtgFirst'+this.mtgId).down('img').src = self.imageRefs.firstIcon;
                $('mtgFirst'+this.mtgId).onclick = function() {
                    self.retrieveDataFromUrl.call(self, 1);
                };
            } else {
                $('mtgFirst'+this.mtgId).down('img').src = self.imageRefs.firstDisabledIcon;
            }

            if (currentPage > 0 && currentPage < pages) {
                $('mtgNext'+this.mtgId).down('img').src = self.imageRefs.nextIcon;
                $('mtgNext'+this.mtgId).onclick = function() {
                    self.retrieveDataFromUrl.call(self, currentPage + 1);
                };
            } else {
                $('mtgNext'+this.mtgId).down('img').src = self.imageRefs.nextDisabledIcon;
            }

            if (currentPage > 1 && currentPage <= pages) {
                $('mtgPrev'+this.mtgId).down('img').src = self.imageRefs.prevIcon;
                $('mtgPrev'+this.mtgId).onclick = function() {
                    self.retrieveDataFromUrl.call(self, currentPage - 1);
                };
            } else {
                $('mtgPrev'+this.mtgId).down('img').src = self.imageRefs.prevDisabledIcon;
            }

            if (currentPage < pages) {
                $('mtgLast'+this.mtgId).down('img').src = self.imageRefs.lastIcon;
                $('mtgLast'+this.mtgId).onclick = function() {
                    self.retrieveDataFromUrl.call(self, self.pager.pages);
                };
            } else {
                $('mtgLast'+this.mtgId).down('img').src = self.imageRefs.lastDisabledIcon;
            }

            var keyHandler = function(event) {
                if (event.keyCode == Event.KEY_RETURN) {
                    var pageNumber = $('mtgPageInput'+self.mtgId).value;
                    if (pageNumber > pages) pageNumber = pages;
                    if (pageNumber < 1) pageNumber = '1';
                    $('mtgPageInput'+self.mtgId).value = pageNumber;
                    self.retrieveDataFromUrl.call(self, pageNumber);
                }
            };

            if (Prototype.Browser.Gecko || Prototype.Browser.Opera ) {
                Event.observe($('mtgPageInput'+this.mtgId), 'keypress', function(event) {
                    keyHandler(event);
                });
            } else {
                Event.observe($('mtgPageInput'+this.mtgId), 'keydown', function(event) {
                    keyHandler(event);
                });
            }
        }
    },

    resize : function() {
        var target = $(this.target);
        var width = this.options.width || (target.getWidth() - this.fullPadding(target,'left') - this.fullPadding(target,'right')) + 'px';
        var height = this.options.height || (target.getHeight() - this.fullPadding(target,'top') - this.fullPadding(target,'bottom')) + 'px';
        this.tableWidth = parseInt(width) - 2;
        var tallerFlg = false;
        if ((parseInt(height) - 2) > this.tableHeight) tallerFlg = true;
        this.tableHeight = parseInt(height) - 2;

        this.tableDiv.setStyle({
            width: this.tableWidth + 'px',
            height: this.tableHeight + 'px'
        });

        if (this.options.title) {
            $('mtgHeaderTitle'+this.mtgId).setStyle({
                width: (this.tableWidth - 6) + 'px'
            });
        }

        this.headerRowDiv.setStyle({
            width: (this.tableWidth) + 'px'
        });

        this.bodyHeight = this.tableHeight - this.headerHeight - 3;
        if (this.options.title) this.bodyHeight = this.bodyHeight - this.headerHeight - 1;
        if (this.options.pager) this.bodyHeight = this.bodyHeight - this.pagerHeight;

        this.bodyDiv.setStyle({
            width: (this.tableWidth) + 'px',
            height: this.bodyHeight + 'px'
        });

        if (this.options.pager) {
            var topPos = this.bodyHeight + this.headerHeight + 3 + ((this.options.title || this.options.onSave) ? this.headerHeight : 0);
            this.pagerDiv.setStyle({
                top: topPos + 'px',
                width: (this.tableWidth - 2) + 'px'
            });
        }

        this.renderedRowsAllowed = Math.floor(this.bodyDiv.clientHeight / this.cellHeight);
        
        if (tallerFlg) {
            var html = this.createTableBody(this.rows);
            this.bodyTable.down('tbody').insert(html);
            this.addKeyBehavior();
            this.keys.addMouseBehavior();
        }
    },

    getValueAt : function(x, y) {
        x = this.getColumnIndex(this.columnModel[x].id);
        return this.rows[y][x];
    },

    setValueAt : function(value, x, y) {
        $('mtgIC'+this.mtgId+'_'+x+','+y).innerHTML = value;
        x = this.getColumnIndex(this.columnModel[x].id);
        this.rows[y][x] = value;
    },

    getColumnIndex : function(id) {
        var index = -1;
        for (var i = 0; i < this.columnModel.length; i++) {
            if (this.columnModel[i].id == id) {
                index = this.columnModel[i].positionIndex;
                break;
            }
        }
        return index;
    },

    getIndexOf : function(id) {
        var i = 0;
        for (; i < this.columnModel.length; i++) {
            if (this.columnModel[i].id == id) break;
        }
        return i;
    },

    getCurrentPosition : function() {
        return [this.keys.xCurrentPos, this.keys.yCurrentPos];
    },

    getCellElementAt : function(x, y) {
        return $('mtgC'+this.mtgId + '_' + x + ',' + y);
    },

    getModifiedRows : function() {
        var result = [];
        for (var i = 0; i < this.modifiedRows.length; i++) {
            var idx = this.modifiedRows[i];
            var row = {};
            for (var j = 0; j < this.columnModel.length; j++) {
                row[this.columnModel[j].id] = this.rows[idx][this.columnModel[j].positionIndex];
            }
            result.push(row);
        }
        return result;
    },

    highlightRow : function(id, value) {
        $$('.mtgRow'+this.mtgId).each(function(row){
            row.removeClassName('focus');
        });

        var index = this.getColumnIndex(id);
        var rowIndex = -1;
        for (var i = 0; i < this.rows.length; i++) {
            if (this.rows[i][index] == value) {
                rowIndex = i;
                break;
            }
        }

        if (rowIndex >= 0) {
            $('mtgRow'+this.mtgId+'_'+rowIndex).addClassName('focus');
        }
    },

    getRow : function(index) {
        var result = {};
        for (var j = 0; j < this.columnModel.length; j++) {
            result[this.columnModel[j].id] = this.rows[index][this.columnModel[j].positionIndex];
        }
        return result;
    },

    clear : function() {
        this.modifiedRows = [];
    }
};