/**
 * jQuery Plasma Plug-in, version 0.2
 * Copyright (c) 2011 Daniel MacDonald dpm@projectatomic.com ALL RIGHTS RESERVED
 * 
 * This is free software; you can redistribute it and/or modify 
 * it under the terms of the GNU Lesser General Public License 
 * as published by the Free Software Foundation; either version 
 * 2.1 of the License, or (at your option) any later version.
 * 
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 * 
 * 
 * This plug-in creates a table and cycles the background colors
 * of the table cell with an HSV Sine transformation. This plug-in
 * can become processor intensive with too many rows or columns,
 * which could cause a browser to react slowly, freeze, or crash.
 * The shift and blur option seem to have the most noticeable
 * effect on the plasma.
 * 
 * VERSION 0.2
 * optimized the delay timer
 * added start, stop, destroy methods
 * added data support for settings to allow on the fly changes
 * 
 * VERSION 0.1
 * initial release, not ready for prime time
 * 
 * ROADMAP:
 * fix bugs
 * create an overlay method
 * work on performance
 */

(function( $ ){

  var settings = {
    rows: 8, // more rows = more processing cycles
    cols: 8, // more cols = more processing cycles
    spacing: 0, // table's cellspacing attribute
    padding: 0, // table's cellpadding attribute
    border: 0, // table's border attribute
    height: '40px', // css height property for each cell
    width: '40px', // css height property for each cell
    html: '', // each cell can contain some html
    shift: 128000, // seems almost any integer is possible
    blur: 1, // less than 1 sharpens the plasma
    r: 1, // red intensity 0-1
    g: 1, // green intensity 0-1
    b: 1, // blue intensity 0-1
    s: 1, // saturation intensity 0-1
    v: 1, // value intensity 0-1
    delay: 100
  };
 
  var $this;
 
  var methods = {

/*
 * init method
 * creates table and sets interval timer for plasma
 */
    init: function(options) {
     
      return this.each(function(){
        $this = $(this);
        if (options) { 
          $.extend(settings, options);
        };
        var table = $('<table/>').attr({cellpadding:settings.padding, cellspacing:settings.spacing, border:settings.border});
        for (i = 0; i < settings.rows; i++) {
          var row = $('<tr/>');
          for (j = 0; j < settings.cols; j++) {
            row.append($('<td/>').css({height:settings.height, width:settings.width}).html(settings.html));
          };
          table.append(row);
        };
        $this.append(table);
        $this.data('plasma', settings);
        methods.plasma_morph();
      });
    },

/*
 * plasma_morph method (private)
 * runs every delay option milliseconds
 * performs Sine wave transformation, cycles HSV colors, sets background color of each table cell
 */   
    plasma_morph: function() {
      var data = $this.data('plasma');
      for(y = 0; y < data.rows; y++) {
        for(x = 0; x < data.cols; x++) {
          var value = Math.sin(methods.distort(x + data.shift, y, 128, 128) / data.cols / data.blur) + Math.sin(methods.distort(x, y + settings.shift / (data.rows - 1) / data.blur, 192, 64) / (data.rows - 1) / data.blur);
          var color = methods.HSVtoRGB(value * 255 / data.blur, data.s, data.v);
          methods.set_cell(x, y, color.r * data.r, color.g * data.g, color.b * data.b);
        };
      };
      settings.shift++;
      $.extend(data, {timeout: window.setTimeout(methods.plasma_morph, data.delay)})
    },
  
/*
 * set_cell method (private)
 * x integer column < settings.cols
 * y integer row < settings.row
 * r, g, b integer 0-255
 */    
    set_cell: function(x, y, r, g, b) {
      /** future development will include overlay functionality which will go here */
      $this.find('tr').eq(y).find('td').eq(x).css('background-color', 'rgb('+r+','+g+','+b+')');
    },

/*
 * HSVtoRGB method (private)
 * h (hue) 0-360
 * s (saturation) 0-1
 * v (value) 0-1
 * returns object with r, g, b properties 0-255
 */    
    HSVtoRGB: function(h,s,v) { 
      var r,g,b,i,f,p,q,t;
      while (h<0) { h += 360; };
      h %= 360;
      s = s > 1 ? 1 : s < 0 ? 0 : s;
      v = v > 1 ? 1 : v < 0 ? 0 : v;
      
      if (s == 0) {
        r = g = b = v;
      } else {
        h /= 60;
        f = h - (i = Math.floor(h));
        p = v * (1 - s);
        q = v * (1 - s * f);
        t = v * (1 - s * (1 - f));
        switch (i) {
          case 0: r = v; g = t; b = p; break;
          case 1: r = q; g = v; b = p; break;
          case 2: r = p; g = v; b = t; break;
          case 3: r = p; g = q; b = v; break;
          case 4: r = t; g = p; b = v; break;
          case 5: r = v; g = p; b = q; break;
        };
      };
      return {
        r: Math.floor(r*255),
        g: Math.floor(g*255),
        b: Math.floor(b*255)
      };
    },
 
/*
 * distort method (private)
 * a, b, c, d numeric, not 0
 */    
    distort : function(a, b, c, d) {
      return Math.sqrt((c-a)*(c-a)+(d-b)*(d-b));
    },

/*
 * start method
 * restarts a stopped animation
 */    
    start : function() {
      return this.each(function(){
        if ($(this).data('plasma')) {
          methods.plasma_morph();
        }
      });
    },

/*
 * stop method
 * clears the timeout to plasma_morph method
 */    
    stop : function() {
      return this.each(function(){
        if ($(this).data('plasma')) {
          window.clearTimeout($(this).data('plasma').timeout);
        }
      });
    },

/*
 * destroy method
 * clears the timeout to plasma_morph method
 * removes stored data
 */    
    destroy : function() {
      return this.each(function(){
        if ($(this).data('plasma')) {
          window.clearTimeout($(this).data('plasma').timeout);
          $(this).removeData('plasma').empty();
        }
      });
    }
  };

/*
 * jQuery plug-in declaration
 */
  $.fn.plasma = function(method) {
    if (methods[method]) {
      return methods[method].apply(this, Array.prototype.slice.call(arguments, 1));
    } else if (typeof method === 'object' || !method) {
      return methods.init.apply(this, arguments);
    } else {
      $.error('Method ' + method + ' does not exist on jQuery.plasma');
    }    
  };

})(jQuery);
