﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using NorthwindViewModel;
using NorthwindDataModel;
using NorthwindDataAccessServices;
using NorthwindBusinessServices;

namespace NorthwindApplicationServices
{
    public class OrderApplicationService
    {

        /// <summary>
        /// Start Order Entry
        /// </summary>
        /// <param name="viewModel"></param>
        /// <returns></returns>
        public OrderViewModel BeginOrderEntry(string customerID)
        {
          
            CustomerDataAccessService customerDataAccessService = new CustomerDataAccessService();         
            Customer customer = customerDataAccessService.GetCustomerInformation(customerID);
       
            OrderViewModel orderViewModel = new OrderViewModel();
            orderViewModel.Customer = customer;

            OrderDataAccessService orderDataAccessService = new OrderDataAccessService();
            orderViewModel.Shippers = orderDataAccessService.GetShippers();

            OrderBusinessService orderBusinessService = new OrderBusinessService();
            orderViewModel.Order = orderBusinessService.InitializeOrderHeader(customer);

            return orderViewModel;

        }


        /// <summary>
        /// Start Order Edit
        /// </summary>
        /// <param name="viewModel"></param>
        /// <returns></returns>
        public OrderViewModel BeginOrderEdit(int orderID)
        {

            OrderDataAccessService orderDataAccessService = new OrderDataAccessService();
            OrderViewModel orderViewModel = new OrderViewModel();

            OrdersCustomer orderCustomer = orderDataAccessService.GetOrder(orderID);
            orderCustomer.Order.OrderTotal = orderDataAccessService.GetOrderTotal(orderID);
            orderCustomer.Order.OrderTotalFormatted = orderCustomer.Order.OrderTotal.ToString("C");

            orderViewModel.Customer = orderCustomer.Customer;
            orderViewModel.Order = orderCustomer.Order;           
            orderViewModel.Shippers = orderDataAccessService.GetShippers();
            orderViewModel.Order.ShipperName = orderCustomer.Shipper.CompanyName;

            return orderViewModel;

        }

        /// <summary>
        /// Get Order Details
        /// </summary>
        /// <param name="viewModel"></param>
        /// <returns></returns>
        public OrderViewModel GetOrderDetails(int orderID)
        {

            OrderDataAccessService orderDataAccessService = new OrderDataAccessService();
            OrderViewModel orderViewModel = new OrderViewModel();

            List<OrderDetailsProducts> orderDetailsProducts = orderDataAccessService.GetOrderDetails(orderID);
            OrdersCustomer orderCustomer = orderDataAccessService.GetOrder(orderID);
            orderViewModel.OrderDetailsProducts = orderDetailsProducts;
            orderViewModel.Order = orderCustomer.Order;
            orderViewModel.Customer = orderCustomer.Customer;
          
            return orderViewModel;

        }

        /// <summary>
        /// Add Order Detail Line Item
        /// </summary>
        /// <param name="orderViewModel"></param>
        /// <returns></returns>
        public OrderViewModel AddOrderDetailLineItem(OrderViewModel orderViewModel)
        {
            try
            {
                OrderBusinessService orderBusinessService = new OrderBusinessService();
                orderBusinessService.ValidateOrderDetail(orderViewModel.OrderDetail, true);
                if (orderBusinessService.ValidationStatus == false)
                {
                    orderViewModel.ReturnMessage = orderBusinessService.ValidationMessage;
                    orderViewModel.ReturnStatus = false;
                    orderViewModel.ValidationErrors = orderBusinessService.ValidationErrors;
                    return orderViewModel;
                }

                OrderDataAccessService orderDataAccessService = new OrderDataAccessService();
                orderDataAccessService.CreateOrderDetailLineItem(orderViewModel.OrderDetail);

                OrderDetailsProducts orderDetailsProducts = orderDataAccessService.GetOrderDetails(
                    orderViewModel.OrderDetail.OrderID, orderViewModel.OrderDetail.ProductID);

                orderViewModel.OrderLineItem = orderDetailsProducts;
                orderViewModel.ReturnStatus = true;

                List<String> returnMessage = new List<String>();
                returnMessage.Add("Order line item has been added at " + DateTime.Now.ToString());
                orderViewModel.ReturnMessage = returnMessage;
             
            }
            catch (Exception ex)
            {
                List<String> returnMessage = new List<String>();
                returnMessage.Add(ex.Message);

                orderViewModel.ReturnMessage = returnMessage;
                orderViewModel.ReturnStatus = false;
            }

            return orderViewModel;

        }

        /// <summary>
        /// Update Order Detail Line Item
        /// </summary>
        /// <param name="orderViewModel"></param>
        /// <returns></returns>
        public OrderViewModel UpdateOrderDetailLineItem(OrderViewModel orderViewModel)
        {
            try
            {
                OrderBusinessService orderBusinessService = new OrderBusinessService();
                orderBusinessService.ValidateOrderDetail(orderViewModel.OrderDetail, false);
                if (orderBusinessService.ValidationStatus == false)
                {
                    orderViewModel.ReturnMessage = orderBusinessService.ValidationMessage;
                    orderViewModel.ReturnStatus = false;
                    orderViewModel.ValidationErrors = orderBusinessService.ValidationErrors;
                    return orderViewModel;
                }

                OrderDataAccessService orderDataAccessService = new OrderDataAccessService();
                orderDataAccessService.UpdateOrderDetailLineItem(orderViewModel.OrderDetail);

                OrderDetailsProducts orderDetailsProducts = orderDataAccessService.GetOrderDetails(
                    orderViewModel.OrderDetail.OrderID, orderViewModel.OrderDetail.ProductID);

                orderViewModel.OrderLineItem = orderDetailsProducts;
                orderViewModel.ReturnStatus = true;

                List<String> returnMessage = new List<String>();
                returnMessage.Add("Order line item has been updated at " + DateTime.Now.ToString());
                orderViewModel.ReturnMessage = returnMessage;

            }
            catch (Exception ex)
            {
                List<String> returnMessage = new List<String>();
                returnMessage.Add(ex.Message);

                orderViewModel.ReturnMessage = returnMessage;
                orderViewModel.ReturnStatus = false;
            }

            return orderViewModel;

        }

     
         /// <summary>
        /// Delete Order Detail Line Item
        /// </summary>
        /// <param name="orderViewModel"></param>
        /// <returns></returns>
        public OrderViewModel DeleteOrderDetailLineItem(OrderViewModel orderViewModel)
        {
            try
            {
                OrderBusinessService orderBusinessService = new OrderBusinessService();

                int productID = orderViewModel.OrderLineItem.OrderDetails.ProductID;
                int orderID = orderViewModel.OrderLineItem.OrderDetails.OrderID;

                OrderDataAccessService orderDataAccessService = new OrderDataAccessService();
                orderDataAccessService.DeleteOrderDetailLineItem(orderID, productID);
              
                orderViewModel.ReturnStatus = true;

                List<String> returnMessage = new List<String>();
                returnMessage.Add(orderViewModel.OrderLineItem.Products.ProductName + " has been delete from this order at " + DateTime.Now.ToString());
                orderViewModel.ReturnMessage = returnMessage;

            }
            catch (Exception ex)
            {
                List<String> returnMessage = new List<String>();
                returnMessage.Add(ex.Message);

                orderViewModel.ReturnMessage = returnMessage;
                orderViewModel.ReturnStatus = false;
            }

            return orderViewModel;

        }

        /// <summary>
        /// Create Order
        /// </summary>
        /// <param name="orderViewModel"></param>
        /// <returns></returns>
        public OrderViewModel CreateOrder(OrderViewModel orderViewModel)
        {
            try
            {

                OrderBusinessService orderBusinessService = new OrderBusinessService();
                orderBusinessService.ValidateOrderHeader(orderViewModel.Order);
                if (orderBusinessService.ValidationStatus == false)
                {
                    orderViewModel.ReturnMessage = orderBusinessService.ValidationMessage;
                    orderViewModel.ReturnStatus = false;
                    orderViewModel.ValidationErrors = orderBusinessService.ValidationErrors;
                    return orderViewModel;
                }

                OrderDataAccessService orderDataAccessService = new OrderDataAccessService();
                orderDataAccessService.CreateOrder(orderViewModel.Order);
                orderViewModel = BeginOrderEdit(orderViewModel.Order.OrderID);
                orderViewModel.Order.RequiredDateFormatted = orderViewModel.Order.RequiredDate.ToShortDateString();

                List<String> returnMessage = new List<String>();
                returnMessage.Add("Order number " + orderViewModel.Order.OrderID.ToString() + " has been created.");

                orderViewModel.ReturnMessage = returnMessage;
                orderViewModel.ReturnStatus = true;

            }
            catch (Exception ex)
            {
                List<String> returnMessage = new List<String>();
                returnMessage.Add(ex.Message);

                orderViewModel.ReturnMessage = returnMessage;
                orderViewModel.ReturnStatus = false;
            }
         
            return orderViewModel;

        }


        /// <summary>
        /// Update Order
        /// </summary>
        /// <param name="orderViewModel"></param>
        /// <returns></returns>
        public OrderViewModel UpdateOrder(OrderViewModel orderViewModel)
        {
            try
            {

                OrderBusinessService orderBusinessService = new OrderBusinessService();
                orderBusinessService.ValidateOrderHeader(orderViewModel.Order);
                if (orderBusinessService.ValidationStatus == false)
                {
                    orderViewModel.ReturnMessage = orderBusinessService.ValidationMessage;
                    orderViewModel.ReturnStatus = false;
                    orderViewModel.ValidationErrors = orderBusinessService.ValidationErrors;
                    return orderViewModel;
                }

                OrderDataAccessService orderDataAccessService = new OrderDataAccessService();
                orderDataAccessService.UpdateOrder(orderViewModel.Order);
                orderViewModel = BeginOrderEdit(orderViewModel.Order.OrderID);
                orderViewModel.Order.RequiredDateFormatted = orderViewModel.Order.RequiredDate.ToShortDateString();

                List<String> returnMessage = new List<String>();
                returnMessage.Add("Order has been updated at " + DateTime.Now.ToString());

                orderViewModel.ReturnMessage = returnMessage;
                orderViewModel.ReturnStatus = true;

            }
            catch (Exception ex)
            {
                List<String> returnMessage = new List<String>();
                returnMessage.Add(ex.Message);

                orderViewModel.ReturnMessage = returnMessage;
                orderViewModel.ReturnStatus = false;
            }

            return orderViewModel;

        }

        /// <summary>
        /// Customer Inquiry
        /// </summary>
        /// <param name="viewModel"></param>
        /// <returns></returns>
        public OrderViewModel OrderInquiry(OrderViewModel viewModel)
        {

            int totalOrders;

            List<String> returnMessage = new List<String>();

            OrderDataAccessService orderDataAccessService = new OrderDataAccessService();

            try
            {
                List<OrdersCustomer> ordersCustomers = orderDataAccessService.OrderInquiry(viewModel.Order,
                    viewModel.Customer,
                    viewModel.CurrentPageNumber, 
                    viewModel.SortExpression, 
                    viewModel.SortDirection,
                    viewModel.PageSize, 
                    out totalOrders);

                viewModel.TotalOrders = totalOrders;
                viewModel.TotalPages = Utilities.CalculateTotalPages(totalOrders, viewModel.PageSize);
                viewModel.OrderCustomer = ordersCustomers;

                if (viewModel.TotalOrders > 0 && viewModel.TotalPages < viewModel.CurrentPageNumber)
                {
                    returnMessage.Add("Please resubmit your request.");
                    viewModel.ReturnMessage = returnMessage;
                    viewModel.ReturnStatus = false;
                }
                else
                {
                    returnMessage.Add(totalOrders.ToString() + " orders found.");
                    viewModel.ReturnMessage = returnMessage;
                    viewModel.ReturnStatus = true;
                }

            }
            catch
            {
                returnMessage.Add("An application error has occurred. Please call technical support.");
                viewModel.ReturnStatus = false;
            }

            viewModel.ReturnMessage = returnMessage;

            return viewModel;

        }


    }

}
